<?php
/**
 * صفحه مدیریت لایسنس افزونه
 */

// Set proper charset headers
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');
mb_http_output('UTF-8');

// شروع session اگر شروع نشده باشد
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/includes/functions.php';

// بررسی لاگین
if (!isLoggedIn()) {
    logSecurityEvent('unauthorized_access_attempt', 'Unauthorized access attempt to license-management.php from IP: ' . getClientIP(), 'high');
    header('Location: 403.php');
    exit;
}
checkSessionTimeout();

$db = new Database();
$conn = $db->getConnection();

// پردازش درخواست‌ها
$message = '';
$message_type = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'add':
                $domain = trim($_POST['domain'] ?? '');
                $license_key = trim($_POST['license_key'] ?? '');
                $expires_at = !empty($_POST['expires_at']) ? $_POST['expires_at'] : null;
                $notes = trim($_POST['notes'] ?? '');
                
                if (empty($domain)) {
                    $message = 'لطفاً دامنه را وارد کنید';
                    $message_type = 'error';
                } else {
                    // پاک کردن www. از ابتدای دامنه
                    $domain = preg_replace('/^www\./', '', $domain);
                    $domain = strtolower(trim($domain));
                    
                    // حذف http:// و https://
                    $domain = preg_replace('/^https?:\/\//', '', $domain);
                    
                    // حذف پورت اگر وجود دارد
                    $domain = preg_replace('/:\d+$/', '', $domain);
                    
                    // تولید license_key خودکار اگر خالی باشد
                    if (empty($license_key)) {
                        $license_key = 'GAV-' . strtoupper(substr(md5($domain . time()), 0, 16));
                    }
                    
                    try {
                        $stmt = $conn->prepare("
                            INSERT INTO plugin_licenses (domain, license_key, expires_at, notes, status) 
                            VALUES (:domain, :license_key, :expires_at, :notes, 'active')
                        ");
                        $stmt->execute([
                            ':domain' => $domain,
                            ':license_key' => $license_key,
                            ':expires_at' => $expires_at ?: null,
                            ':notes' => $notes
                        ]);
                        $message = 'لایسنس با موفقیت اضافه شد';
                        $message_type = 'success';
                    } catch (PDOException $e) {
                        if ($e->getCode() == 23000) {
                            $message = 'این دامنه یا لایسنس قبلاً ثبت شده است';
                        } else {
                            $message = 'خطا در افزودن لایسنس: ' . $e->getMessage();
                        }
                        $message_type = 'error';
                    }
                }
                break;
                
            case 'delete':
                $id = intval($_POST['id'] ?? 0);
                if ($id > 0) {
                    try {
                        $stmt = $conn->prepare("DELETE FROM plugin_licenses WHERE id = :id");
                        $stmt->execute([':id' => $id]);
                        $message = 'لایسنس با موفقیت حذف شد';
                        $message_type = 'success';
                    } catch (PDOException $e) {
                        $message = 'خطا در حذف لایسنس: ' . $e->getMessage();
                        $message_type = 'error';
                    }
                }
                break;
                
            case 'toggle_status':
                $id = intval($_POST['id'] ?? 0);
                $new_status = $_POST['new_status'] ?? 'inactive';
                if ($id > 0) {
                    try {
                        $stmt = $conn->prepare("UPDATE plugin_licenses SET status = :status WHERE id = :id");
                        $stmt->execute([
                            ':id' => $id,
                            ':status' => $new_status
                        ]);
                        $message = 'وضعیت لایسنس با موفقیت تغییر کرد';
                        $message_type = 'success';
                    } catch (PDOException $e) {
                        $message = 'خطا در تغییر وضعیت: ' . $e->getMessage();
                        $message_type = 'error';
                    }
                }
                break;
        }
    }
}

// بررسی وجود جدول و ایجاد آن در صورت نیاز
try {
    $tableCheck = $conn->query("SHOW TABLES LIKE 'plugin_licenses'");
    if ($tableCheck->rowCount() == 0) {
        // جدول وجود ندارد، آن را ایجاد کن
        $createTableSQL = file_get_contents(__DIR__ . '/sql/create-license-table.sql');
        if ($createTableSQL) {
            $createTableSQL = preg_replace('/--.*$/m', '', $createTableSQL);
            $createTableSQL = preg_replace('/\/\*.*?\*\//s', '', $createTableSQL);
            $statements = array_filter(array_map('trim', explode(';', $createTableSQL)));
            foreach ($statements as $statement) {
                if (!empty($statement) && strlen($statement) > 10) {
                    try {
                        $conn->exec($statement);
                    } catch (PDOException $e) {
                        error_log("Error creating license table: " . $e->getMessage());
                    }
                }
            }
        }
    }
} catch (PDOException $e) {
    error_log("Error checking license table: " . $e->getMessage());
}

// دریافت لیست لایسنس‌ها
$licenses = [];
try {
    $stmt = $conn->query("SELECT * FROM plugin_licenses ORDER BY created_at DESC");
    $licenses = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    error_log("Error fetching licenses: " . $e->getMessage());
    // اگر جدول وجود ندارد، پیام خطا نمایش بده
    if (strpos($e->getMessage(), "doesn't exist") !== false || strpos($e->getMessage(), "Unknown table") !== false) {
        $message = 'جدول لایسنس وجود ندارد. لطفاً install.php را اجرا کنید.';
        $message_type = 'error';
    }
}

// دریافت آمار
$stats = [
    'total' => count($licenses),
    'active' => count(array_filter($licenses, fn($l) => $l['status'] === 'active')),
    'inactive' => count(array_filter($licenses, fn($l) => $l['status'] === 'inactive')),
    'suspended' => count(array_filter($licenses, fn($l) => $l['status'] === 'suspended'))
];

$page_title = 'مدیریت لایسنس';
$current_page = 'license-management.php';
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>مدیریت لایسنس - پنل مدیریت گواهیتو</title>
    <link rel="stylesheet" href="assets/admin-style.css">
    <link rel="icon" type="image/png" href="assets/images/logo.png">
    <?php include 'includes/admin-styles.php'; ?>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    <?php include 'includes/sidebar.php'; ?>
    
    <!-- Sidebar Overlay for Mobile -->
    <div class="sidebar-overlay" id="sidebarOverlay"></div>

    <!-- Main Content -->
    <div class="main-content" id="mainContent">
        <div class="page-header">
            <h1 class="page-title">🔐 مدیریت لایسنس افزونه</h1>
            <p class="page-description">مدیریت و کنترل دسترسی افزونه گواهیتو</p>
        </div>

        <?php if ($message): ?>
            <div class="alert alert-<?php echo $message_type === 'success' ? 'success' : 'danger'; ?>">
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>

        <!-- آمار -->
        <div class="grid grid-cols-1 grid-cols-2 grid-cols-4" style="margin-bottom: 2rem;">
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['total']; ?></div>
                <div class="stats-label">کل لایسنس‌ها</div>
            </div>
            <div class="stats-card">
                <div class="stats-number" style="color: #10b981;"><?php echo $stats['active']; ?></div>
                <div class="stats-label">فعال</div>
            </div>
            <div class="stats-card">
                <div class="stats-number" style="color: #f59e0b;"><?php echo $stats['inactive']; ?></div>
                <div class="stats-label">غیرفعال</div>
            </div>
            <div class="stats-card">
                <div class="stats-number" style="color: #ef4444;"><?php echo $stats['suspended']; ?></div>
                <div class="stats-label">معلق</div>
            </div>
        </div>

        <!-- فرم افزودن لایسنس -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">➕ افزودن لایسنس جدید</h2>
            </div>
            <div class="card-body">
                <form method="POST" class="license-form">
                    <input type="hidden" name="action" value="add">
                    <div class="form-group">
                        <label for="domain" class="form-label">
                            <span class="label-icon">🌐</span>
                            <span class="label-text">دامنه <span class="required">*</span></span>
                        </label>
                        <input type="text" id="domain" name="domain" required 
                               placeholder="example.com" class="form-control">
                        <small class="form-hint">دامنه بدون http:// یا https://</small>
                    </div>
                    <div class="form-group">
                        <label for="license_key" class="form-label">
                            <span class="label-icon">🔑</span>
                            <span class="label-text">کلید لایسنس</span>
                        </label>
                        <input type="text" id="license_key" name="license_key" 
                               placeholder="خالی بگذارید تا خودکار تولید شود" class="form-control">
                        <small class="form-hint">اگر خالی بگذارید، کلید به صورت خودکار تولید می‌شود</small>
                    </div>
                    <div class="form-group">
                        <label for="expires_at" class="form-label">
                            <span class="label-icon">📅</span>
                            <span class="label-text">تاریخ انقضا <span class="optional">(اختیاری)</span></span>
                        </label>
                        <input type="datetime-local" id="expires_at" name="expires_at" class="form-control">
                        <small class="form-hint">در صورت خالی بودن، لایسنس نامحدود خواهد بود</small>
                    </div>
                    <div class="form-group full-width">
                        <label for="notes" class="form-label">
                            <span class="label-icon">📝</span>
                            <span class="label-text">یادداشت</span>
                        </label>
                        <textarea id="notes" name="notes" rows="4" class="form-control" 
                                  placeholder="یادداشت اختیاری..."></textarea>
                        <small class="form-hint">یادداشت‌های اضافی درباره این لایسنس</small>
                    </div>
                    <div class="form-group full-width">
                        <button type="submit" class="btn btn-primary btn-lg">
                            <span class="btn-icon">✅</span>
                            <span>افزودن لایسنس</span>
                        </button>
                    </div>
                </form>
            </div>
        </div>

        <!-- لیست لایسنس‌ها -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">📋 لیست لایسنس‌ها</h2>
            </div>
            <div class="card-body">
                <?php if (empty($licenses)): ?>
                    <p style="color: var(--dark-400); text-align: center; padding: 2rem;">هنوز لایسنسی ثبت نشده است</p>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>دامنه</th>
                                <th>کلید لایسنس</th>
                                <th>وضعیت</th>
                                <th>تاریخ انقضا</th>
                                <th>تاریخ ایجاد</th>
                                <th>عملیات</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($licenses as $license): ?>
                                <tr>
                                    <td><?php echo $license['id']; ?></td>
                                    <td><strong><?php echo htmlspecialchars($license['domain']); ?></strong></td>
                                    <td>
                                        <code class="license-key"><?php echo htmlspecialchars($license['license_key']); ?></code>
                                        <button class="btn-copy" onclick="copyToClipboard('<?php echo htmlspecialchars($license['license_key']); ?>')" title="کپی">
                                            📋
                                        </button>
                                    </td>
                                    <td>
                                        <span class="badge badge-<?php 
                                            echo $license['status'] === 'active' ? 'success' : 
                                                ($license['status'] === 'inactive' ? 'warning' : 'danger'); 
                                        ?>">
                                            <?php 
                                            echo $license['status'] === 'active' ? 'فعال' : 
                                                ($license['status'] === 'inactive' ? 'غیرفعال' : 'معلق'); 
                                            ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php echo $license['expires_at'] ? 
                                            date('Y/m/d H:i', strtotime($license['expires_at'])) : 
                                            'نامحدود'; ?>
                                    </td>
                                    <td><?php echo date('Y/m/d H:i', strtotime($license['created_at'])); ?></td>
                                    <td>
                                        <div class="action-buttons">
                                            <form method="POST" style="display: inline;">
                                                <input type="hidden" name="action" value="toggle_status">
                                                <input type="hidden" name="id" value="<?php echo $license['id']; ?>">
                                                <input type="hidden" name="new_status" 
                                                       value="<?php echo $license['status'] === 'active' ? 'inactive' : 'active'; ?>">
                                                <button type="submit" class="btn btn-sm btn-<?php 
                                                    echo $license['status'] === 'active' ? 'warning' : 'success'; 
                                                ?>" title="<?php 
                                                    echo $license['status'] === 'active' ? 'غیرفعال کردن' : 'فعال کردن'; 
                                                ?>">
                                                    <?php echo $license['status'] === 'active' ? '⏸️' : '✅'; ?>
                                                </button>
                                            </form>
                                            <form method="POST" style="display: inline;" 
                                                  onsubmit="return confirm('آیا از حذف این لایسنس مطمئن هستید؟');">
                                                <input type="hidden" name="action" value="delete">
                                                <input type="hidden" name="id" value="<?php echo $license['id']; ?>">
                                                <button type="submit" class="btn btn-sm btn-danger" title="حذف">🗑️</button>
                                            </form>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <style>
    /* استایل فرم لایسنس */
    .license-form {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
        gap: 1.5rem;
        padding: 1.5rem;
    }
    
    .license-form .form-group {
        display: flex;
        flex-direction: column;
        gap: 0.5rem;
    }
    
    .license-form .form-group.full-width {
        grid-column: 1 / -1;
    }
    
    .license-form .form-label {
        display: flex;
        align-items: center;
        gap: 0.5rem;
        font-weight: 600;
        color: var(--text-primary, #fff);
        margin-bottom: 0.25rem;
        font-size: 0.95rem;
    }
    
    .license-form .label-icon {
        font-size: 1.2rem;
        display: flex;
        align-items: center;
        justify-content: center;
        width: 28px;
        height: 28px;
        background: linear-gradient(135deg, rgba(197, 145, 44, 0.2), rgba(4, 28, 63, 0.2));
        border-radius: 8px;
        flex-shrink: 0;
    }
    
    .license-form .label-text {
        flex: 1;
    }
    
    .license-form .required {
        color: #ef4444;
        font-weight: 700;
        margin-right: 2px;
    }
    
    .license-form .optional {
        color: var(--text-secondary, rgba(255, 255, 255, 0.6));
        font-weight: 400;
        font-size: 0.85em;
    }
    
    .license-form .form-control {
        width: 100%;
        padding: 0.75rem 1rem;
        background: rgba(255, 255, 255, 0.05);
        border: 2px solid rgba(255, 255, 255, 0.1);
        border-radius: 12px;
        color: var(--text-primary, #fff);
        font-size: 0.95rem;
        transition: all 0.3s ease;
        backdrop-filter: blur(10px);
    }
    
    .license-form .form-control:focus {
        outline: none;
        border-color: rgba(197, 145, 44, 0.6);
        background: rgba(255, 255, 255, 0.08);
        box-shadow: 0 0 0 3px rgba(197, 145, 44, 0.1);
    }
    
    .license-form .form-control::placeholder {
        color: rgba(255, 255, 255, 0.4);
    }
    
    .license-form textarea.form-control {
        resize: vertical;
        min-height: 100px;
        font-family: inherit;
    }
    
    .license-form .form-hint {
        color: var(--text-secondary, rgba(255, 255, 255, 0.6));
        font-size: 0.85rem;
        margin-top: 0.25rem;
        display: flex;
        align-items: center;
        gap: 0.25rem;
    }
    
    .license-form .form-hint::before {
        content: "💡";
        font-size: 0.9rem;
    }
    
    .license-form .btn-lg {
        padding: 1rem 2rem;
        font-size: 1.05rem;
        font-weight: 600;
        border-radius: 12px;
        display: flex;
        align-items: center;
        justify-content: center;
        gap: 0.75rem;
        background: linear-gradient(135deg, #c5912c, #041c3f);
        border: none;
        color: #fff;
        cursor: pointer;
        transition: all 0.3s ease;
        box-shadow: 0 4px 15px rgba(197, 145, 44, 0.3);
        width: 100%;
        max-width: 300px;
        margin: 0 auto;
    }
    
    .license-form .btn-lg:hover {
        transform: translateY(-2px);
        box-shadow: 0 6px 20px rgba(197, 145, 44, 0.4);
    }
    
    .license-form .btn-lg:active {
        transform: translateY(0);
    }
    
    .license-form .btn-icon {
        font-size: 1.2rem;
    }
    
    /* استایل کلید لایسنس در جدول */
    .license-key {
        background: rgba(197, 145, 44, 0.1);
        padding: 4px 8px;
        border-radius: 4px;
        font-family: monospace;
        font-size: 0.9em;
    }
    .btn-copy {
        background: none;
        border: none;
        cursor: pointer;
        margin-left: 8px;
        font-size: 1.1em;
        padding: 2px 6px;
        transition: all 0.2s ease;
    }
    .btn-copy:hover {
        opacity: 0.7;
        transform: scale(1.1);
    }
    .action-buttons {
        display: flex;
        gap: 5px;
    }
    
    /* Responsive */
    @media (max-width: 768px) {
        .license-form {
            grid-template-columns: 1fr;
            padding: 1rem;
            gap: 1rem;
        }
        
        .license-form .form-label {
            font-size: 0.9rem;
        }
        
        .license-form .label-icon {
            width: 24px;
            height: 24px;
            font-size: 1rem;
        }
    }
    </style>

<script>
function copyToClipboard(text) {
    navigator.clipboard.writeText(text).then(function() {
        alert('کلید لایسنس کپی شد!');
    }, function() {
        alert('خطا در کپی کردن');
    });
}
</script>

<script>
// Mobile menu toggle
document.addEventListener('DOMContentLoaded', function() {
    const mobileMenuToggle = document.getElementById('mobileMenuToggle');
    const sidebar = document.getElementById('sidebar');
    
    if (mobileMenuToggle && sidebar) {
        mobileMenuToggle.addEventListener('click', function() {
            sidebar.classList.toggle('active');
        });
    }
    
    const sidebarClose = document.getElementById('sidebarClose');
    if (sidebarClose && sidebar) {
        sidebarClose.addEventListener('click', function() {
            sidebar.classList.remove('active');
        });
    }
});
</script>
</body>
</html>

