<?php
// Set proper charset headers
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');
mb_http_output('UTF-8');

// Start session
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once 'config/database.php';
require_once 'includes/functions.php';

// Check authentication
if (!isLoggedIn()) {
    logSecurityEvent('unauthorized_access_attempt', 'Unauthorized access attempt to feature-suggestions.php from IP: ' . getClientIP(), 'high');
    header('Location: 403.php');
    exit;
}
checkSessionTimeout();

$message = '';
$error = '';

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        switch ($action) {
            case 'create':
                $title = sanitizeInput($_POST['title'] ?? '');
                $description = sanitizeInput($_POST['description'] ?? '');
                $isActive = isset($_POST['is_active']) ? 1 : 0;
                $isFeatured = isset($_POST['is_featured']) ? 1 : 0;
                $displayOrder = (int)($_POST['display_order'] ?? 0);
                $disableVoteRemoval = isset($_POST['disable_vote_removal']) ? 1 : 0;
                
                // Get admin ID from session
                $adminId = null;
                if (isset($_SESSION['admin_id'])) {
                    $sessionAdminId = (int)$_SESSION['admin_id'];
                    
                    // Verify admin ID exists in database
                    $checkAdminQuery = "SELECT id FROM admin_users WHERE id = ? LIMIT 1";
                    $checkAdminStmt = $conn->prepare($checkAdminQuery);
                    $checkAdminStmt->execute([$sessionAdminId]);
                    $adminUser = $checkAdminStmt->fetch(PDO::FETCH_ASSOC);
                    
                    if ($adminUser) {
                        $adminId = $adminUser['id'];
                    } else {
                        // Admin ID doesn't exist in database, try to get first admin
                        $firstAdminQuery = "SELECT id FROM admin_users WHERE is_active = 1 LIMIT 1";
                        $firstAdminStmt = $conn->query($firstAdminQuery);
                        $firstAdmin = $firstAdminStmt->fetch(PDO::FETCH_ASSOC);
                        if ($firstAdmin) {
                            $adminId = $firstAdmin['id'];
                        }
                        // If no admin found, $adminId remains null (allowed by FK constraint)
                    }
                }
                
                if (empty($title) || empty($description)) {
                    $error = 'عنوان و توضیحات الزامی است';
                } else {
                    // Check if disable_vote_removal column exists
                    try {
                        $checkColumnQuery = "SHOW COLUMNS FROM public_feature_suggestions LIKE 'disable_vote_removal'";
                        $checkColumnStmt = $conn->query($checkColumnQuery);
                        $hasColumn = $checkColumnStmt->rowCount() > 0;
                        $checkColumnStmt->closeCursor();
                    } catch (Exception $e) {
                        $hasColumn = false;
                    }
                    
                    if ($hasColumn) {
                        $insertQuery = "INSERT INTO public_feature_suggestions (title, description, is_active, is_featured, display_order, disable_vote_removal, created_by) VALUES (?, ?, ?, ?, ?, ?, ?)";
                        $insertStmt = $conn->prepare($insertQuery);
                        $insertStmt->execute([$title, $description, $isActive, $isFeatured, $displayOrder, $disableVoteRemoval, $adminId]);
                    } else {
                        $insertQuery = "INSERT INTO public_feature_suggestions (title, description, is_active, is_featured, display_order, created_by) VALUES (?, ?, ?, ?, ?, ?)";
                        $insertStmt = $conn->prepare($insertQuery);
                        $insertStmt->execute([$title, $description, $isActive, $isFeatured, $displayOrder, $adminId]);
                    }
                    
                    $message = "پیشنهاد با موفقیت ایجاد شد";
                    logSecurityEvent('feature_suggestion_created', 'Suggestion created: ' . $title . ' by admin ID: ' . ($adminId ?? 'NULL'), 'low');
                }
                break;
                
            case 'update':
                $id = (int)($_POST['id'] ?? 0);
                $title = sanitizeInput($_POST['title'] ?? '');
                $description = sanitizeInput($_POST['description'] ?? '');
                $isActive = isset($_POST['is_active']) ? 1 : 0;
                $isFeatured = isset($_POST['is_featured']) ? 1 : 0;
                $displayOrder = (int)($_POST['display_order'] ?? 0);
                $disableVoteRemoval = isset($_POST['disable_vote_removal']) ? 1 : 0;
                
                if ($id && !empty($title) && !empty($description)) {
                    // Check if disable_vote_removal column exists
                    try {
                        $checkColumnQuery = "SHOW COLUMNS FROM public_feature_suggestions LIKE 'disable_vote_removal'";
                        $checkColumnStmt = $conn->query($checkColumnQuery);
                        $hasColumn = $checkColumnStmt->rowCount() > 0;
                        $checkColumnStmt->closeCursor();
                    } catch (Exception $e) {
                        $hasColumn = false;
                    }
                    
                    if ($hasColumn) {
                        $updateQuery = "UPDATE public_feature_suggestions SET title = ?, description = ?, is_active = ?, is_featured = ?, display_order = ?, disable_vote_removal = ?, updated_at = NOW() WHERE id = ?";
                        $updateStmt = $conn->prepare($updateQuery);
                        $updateStmt->execute([$title, $description, $isActive, $isFeatured, $displayOrder, $disableVoteRemoval, $id]);
                    } else {
                        $updateQuery = "UPDATE public_feature_suggestions SET title = ?, description = ?, is_active = ?, is_featured = ?, display_order = ?, updated_at = NOW() WHERE id = ?";
                        $updateStmt = $conn->prepare($updateQuery);
                        $updateStmt->execute([$title, $description, $isActive, $isFeatured, $displayOrder, $id]);
                    }
                    
                    $message = "پیشنهاد با موفقیت بروزرسانی شد";
                    logSecurityEvent('feature_suggestion_updated', 'Suggestion ID: ' . $id, 'low');
                } else {
                    $error = 'اطلاعات ناقص است';
                }
                break;
                
            case 'delete':
                $id = (int)($_POST['id'] ?? 0);
                
                if ($id) {
                    // Delete votes first (CASCADE should handle this, but being explicit)
                    $deleteVotesQuery = "DELETE FROM feature_votes WHERE suggestion_id = ?";
                    $deleteVotesStmt = $conn->prepare($deleteVotesQuery);
                    $deleteVotesStmt->execute([$id]);
                    
                    // Delete suggestion
                    $deleteQuery = "DELETE FROM public_feature_suggestions WHERE id = ?";
                    $deleteStmt = $conn->prepare($deleteQuery);
                    $deleteStmt->execute([$id]);
                    
                    $message = "پیشنهاد با موفقیت حذف شد";
                    logSecurityEvent('feature_suggestion_deleted', 'Suggestion ID: ' . $id, 'low');
                }
                break;
                
            case 'toggle_status':
                $id = (int)($_POST['id'] ?? 0);
                
                if ($id) {
                    $toggleQuery = "UPDATE public_feature_suggestions SET is_active = NOT is_active, updated_at = NOW() WHERE id = ?";
                    $toggleStmt = $conn->prepare($toggleQuery);
                    $toggleStmt->execute([$id]);
                    
                    $message = "وضعیت پیشنهاد تغییر کرد";
                }
                break;
        }
    } catch (Exception $e) {
        error_log("Feature suggestions error: " . $e->getMessage());
        $error = "خطا در پردازش درخواست: " . $e->getMessage();
    }
}

// Get all suggestions with vote counts
try {
    $db = new Database();
    $conn = $db->getConnection();
    
    // Check if disable_vote_removal column exists
    try {
        $checkColumnQuery = "SHOW COLUMNS FROM public_feature_suggestions LIKE 'disable_vote_removal'";
        $checkColumnStmt = $conn->query($checkColumnQuery);
        $hasDisableVoteRemovalColumn = $checkColumnStmt->rowCount() > 0;
        $checkColumnStmt->closeCursor();
    } catch (Exception $e) {
        $hasDisableVoteRemovalColumn = false;
    }
    
    if ($hasDisableVoteRemovalColumn) {
        $query = "SELECT 
                    ps.id,
                    ps.title,
                    ps.description,
                    ps.is_active,
                    ps.is_featured,
                    ps.display_order,
                    ps.disable_vote_removal,
                    COUNT(fv.id) as vote_count,
                    ps.created_at,
                    ps.updated_at
                  FROM public_feature_suggestions ps
                  LEFT JOIN feature_votes fv ON ps.id = fv.suggestion_id
                  GROUP BY ps.id, ps.title, ps.description, ps.is_active, ps.is_featured, ps.display_order, ps.disable_vote_removal, ps.created_at, ps.updated_at
                  ORDER BY ps.is_featured DESC, ps.display_order ASC, vote_count DESC, ps.created_at DESC";
    } else {
        $query = "SELECT 
                    ps.id,
                    ps.title,
                    ps.description,
                    ps.is_active,
                    ps.is_featured,
                    ps.display_order,
                    0 as disable_vote_removal,
                    COUNT(fv.id) as vote_count,
                    ps.created_at,
                    ps.updated_at
                  FROM public_feature_suggestions ps
                  LEFT JOIN feature_votes fv ON ps.id = fv.suggestion_id
                  GROUP BY ps.id, ps.title, ps.description, ps.is_active, ps.is_featured, ps.display_order, ps.created_at, ps.updated_at
                  ORDER BY ps.is_featured DESC, ps.display_order ASC, vote_count DESC, ps.created_at DESC";
    }
    
    $stmt = $conn->prepare($query);
    $stmt->execute();
    $suggestions = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get statistics
    $statsQuery = "SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN is_active = 1 THEN 1 ELSE 0 END) as active_count,
        SUM(CASE WHEN is_featured = 1 THEN 1 ELSE 0 END) as featured_count,
        (SELECT COUNT(*) FROM feature_votes) as total_votes
        FROM public_feature_suggestions";
    $statsStmt = $conn->query($statsQuery);
    $stats = $statsStmt->fetch(PDO::FETCH_ASSOC);
    
} catch (Exception $e) {
    error_log("Error fetching suggestions: " . $e->getMessage());
    $suggestions = [];
    $stats = ['total' => 0, 'active_count' => 0, 'featured_count' => 0, 'total_votes' => 0];
}
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>پیشنهادات عمومی - پنل مدیریت گواهیتو</title>
    <link rel="stylesheet" href="assets/admin-style.css">
    <link rel="icon" type="image/png" href="assets/images/logo.png">
    <?php include 'includes/admin-styles.php'; ?>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    <?php include 'includes/sidebar.php'; ?>
    
    <!-- Sidebar Overlay for Mobile -->
    <div class="sidebar-overlay" id="sidebarOverlay"></div>

    <!-- Main Content -->
    <div class="main-content" id="mainContent">
        <div class="page-header">
            <h1 class="page-title">💡 پیشنهادات عمومی</h1>
            <p class="page-description">مدیریت پیشنهادات عمومی برای رای‌دهی کاربران</p>
        </div>

        <?php if ($error): ?>
            <div class="alert alert-danger">
                <?php echo htmlspecialchars($error); ?>
            </div>
        <?php endif; ?>

        <?php if ($message): ?>
            <div class="alert alert-success">
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>

        <!-- Statistics -->
        <div class="grid grid-cols-1 grid-cols-2 grid-cols-3 grid-cols-4" style="margin-bottom: 2rem;">
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['total']; ?></div>
                <div class="stats-label">کل پیشنهادات</div>
            </div>
            
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['active_count']; ?></div>
                <div class="stats-label">پیشنهادات فعال</div>
            </div>
            
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['featured_count']; ?></div>
                <div class="stats-label">پیشنهادات ویژه</div>
            </div>
            
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['total_votes']; ?></div>
                <div class="stats-label">کل رای‌ها</div>
            </div>
        </div>

        <!-- Create New Suggestion Button -->
        <div style="margin-bottom: 1.5rem;">
            <button type="button" class="btn btn-primary" onclick="resetForm(); showSuggestionModal();">
                ➕ ایجاد پیشنهاد جدید
            </button>
        </div>

        <!-- Suggestions List -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">📋 لیست پیشنهادات</h2>
            </div>
            
            <?php if (!empty($suggestions)): ?>
                <div class="table-container">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>عنوان</th>
                                <th>توضیحات</th>
                                <th>وضعیت</th>
                                <th>ویژه</th>
                                <th>ترتیب</th>
                                <th>رای‌ها</th>
                                <th>تاریخ ایجاد</th>
                                <th>عملیات</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($suggestions as $suggestion): ?>
                                <tr style="<?php echo $suggestion['is_featured'] ? 'background: linear-gradient(90deg, rgba(197, 145, 44, 0.1), transparent); border-left: 3px solid #c5912c;' : ''; ?>">
                                    <td>
                                        <strong style="<?php echo $suggestion['is_featured'] ? 'color: #c5912c; font-size: 1.1em;' : ''; ?>">
                                            <?php if ($suggestion['is_featured']): ?>
                                                ⭐ 
                                            <?php endif; ?>
                                            <?php echo htmlspecialchars($suggestion['title']); ?>
                                        </strong>
                                    </td>
                                    <td>
                                        <small style="color: var(--dark-400);">
                                            <?php echo htmlspecialchars(mb_substr($suggestion['description'], 0, 100)) . (mb_strlen($suggestion['description']) > 100 ? '...' : ''); ?>
                                        </small>
                                    </td>
                                    <td>
                                        <span class="badge <?php echo $suggestion['is_active'] ? 'badge-success' : 'badge-secondary'; ?>">
                                            <?php echo $suggestion['is_active'] ? 'فعال' : 'غیرفعال'; ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php if ($suggestion['is_featured']): ?>
                                            <span class="badge badge-warning" style="background: linear-gradient(135deg, #c5912c, #d4a550); color: #fff; font-weight: bold; padding: 0.5rem 1rem; border-radius: 0.5rem; box-shadow: 0 2px 8px rgba(197, 145, 44, 0.3);">
                                                ⭐ ویژه
                                            </span>
                                        <?php else: ?>
                                            <span style="color: var(--dark-400);">عادی</span>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo $suggestion['display_order']; ?></td>
                                    <td>
                                        <span class="badge badge-info">
                                            👍 <?php echo $suggestion['vote_count']; ?>
                                        </span>
                                    </td>
                                    <td><?php echo formatDate($suggestion['created_at']); ?></td>
                                    <td>
                                        <div style="display: flex; gap: 0.5rem; flex-wrap: wrap;">
                                            <button type="button" class="btn btn-warning btn-sm" onclick="editSuggestion(<?php echo htmlspecialchars(json_encode($suggestion)); ?>)">
                                                ویرایش
                                            </button>
                                            <form method="POST" style="display: inline;" onsubmit="return confirm('آیا از تغییر وضعیت این پیشنهاد مطمئن هستید؟');">
                                                <input type="hidden" name="action" value="toggle_status">
                                                <input type="hidden" name="id" value="<?php echo $suggestion['id']; ?>">
                                                <button type="submit" class="btn btn-<?php echo $suggestion['is_active'] ? 'warning' : 'success'; ?> btn-sm" title="<?php echo $suggestion['is_active'] ? 'غیرفعال کردن' : 'فعال کردن'; ?>">
                                                    <?php echo $suggestion['is_active'] ? '👁️‍🗨️' : '👁️'; ?>
                                                </button>
                                            </form>
                                            <form method="POST" style="display: inline;" onsubmit="return confirm('آیا از حذف این پیشنهاد و تمام رای‌های آن مطمئن هستید؟');">
                                                <input type="hidden" name="action" value="delete">
                                                <input type="hidden" name="id" value="<?php echo $suggestion['id']; ?>">
                                                <button type="submit" class="btn btn-danger btn-sm" title="حذف">
                                                    🗑️
                                                </button>
                                            </form>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php else: ?>
                <p style="color: var(--dark-400); text-align: center; padding: 2rem;">هیچ پیشنهادی ثبت نشده است</p>
            <?php endif; ?>
        </div>
    </div>

    <!-- Create/Edit Modal -->
    <div id="suggestionModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 1000;">
        <div style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); background: var(--dark-800); padding: 2rem; border-radius: 1rem; min-width: 500px; max-width: 90%; box-shadow: 0 20px 40px rgba(0,0,0,0.3);">
            <h3 style="margin-bottom: 1rem; color: var(--primary-400);" id="modalTitle">ایجاد پیشنهاد جدید</h3>
            
            <form method="POST" action="" id="suggestionForm">
                <input type="hidden" name="action" id="formAction" value="create">
                <input type="hidden" name="id" id="suggestionId">
                
                <div class="form-group">
                    <label for="title" class="form-label">عنوان *</label>
                    <input type="text" class="form-input" id="title" name="title" required maxlength="200" placeholder="عنوان پیشنهاد">
                </div>
                
                <div class="form-group">
                    <label for="description" class="form-label">توضیحات *</label>
                    <textarea class="form-input" id="description" name="description" rows="5" required placeholder="توضیحات کامل پیشنهاد"></textarea>
                </div>
                
                <div style="display: grid; grid-template-columns: repeat(3, 1fr); gap: 1.5rem; margin-top: 1rem;">
                    <div class="form-group">
                        <label for="display_order" class="form-label">ترتیب نمایش</label>
                        <input type="number" class="form-input" id="display_order" name="display_order" value="0" min="0">
                    </div>
                    
                    <div class="form-group">
                        <div style="background: linear-gradient(135deg, rgba(197, 145, 44, 0.1), rgba(4, 28, 63, 0.2)); border: 1px solid rgba(197, 145, 44, 0.3); border-radius: 0.75rem; padding: 1rem; transition: all 0.3s ease;">
                            <label class="form-label" style="display: block; margin-bottom: 0.75rem; color: rgba(255, 255, 255, 0.9); font-weight: 600;">📊 وضعیت</label>
                            <label style="display: flex; align-items: center; gap: 0.75rem; cursor: pointer; padding: 0.5rem; border-radius: 0.5rem; transition: background 0.2s ease;" 
                                   onmouseover="this.style.background='rgba(197, 145, 44, 0.15)'" 
                                   onmouseout="this.style.background='transparent'">
                                <input type="checkbox" id="is_active" name="is_active" checked 
                                       style="width: 20px; height: 20px; cursor: pointer; accent-color: #c5912c; border-radius: 4px;">
                                <span style="color: rgba(255, 255, 255, 0.85); font-weight: 500; user-select: none;">فعال</span>
                            </label>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <div style="background: linear-gradient(135deg, rgba(197, 145, 44, 0.1), rgba(4, 28, 63, 0.2)); border: 1px solid rgba(197, 145, 44, 0.3); border-radius: 0.75rem; padding: 1rem; transition: all 0.3s ease;">
                            <label class="form-label" style="display: block; margin-bottom: 0.75rem; color: rgba(255, 255, 255, 0.9); font-weight: 600;">⭐ ویژه</label>
                            <label style="display: flex; align-items: center; gap: 0.75rem; cursor: pointer; padding: 0.5rem; border-radius: 0.5rem; transition: background 0.2s ease;" 
                                   onmouseover="this.style.background='rgba(197, 145, 44, 0.15)'" 
                                   onmouseout="this.style.background='transparent'">
                                <input type="checkbox" id="is_featured" name="is_featured" 
                                       style="width: 20px; height: 20px; cursor: pointer; accent-color: #c5912c; border-radius: 4px;">
                                <span style="color: rgba(255, 255, 255, 0.85); font-weight: 500; user-select: none;">ویژه</span>
                            </label>
                        </div>
                    </div>
                </div>
                
                <div class="form-group" style="margin-top: 1.5rem;">
                    <div style="background: linear-gradient(135deg, rgba(220, 38, 38, 0.1), rgba(4, 28, 63, 0.2)); border: 1px solid rgba(220, 38, 38, 0.3); border-radius: 0.75rem; padding: 1.25rem; transition: all 0.3s ease;">
                        <label class="form-label" style="display: block; margin-bottom: 1rem; color: rgba(255, 255, 255, 0.9); font-weight: 600; font-size: 1.05rem;">
                            🔒 غیرفعال کردن حذف رای
                        </label>
                        <label style="display: flex; align-items: flex-start; gap: 0.75rem; cursor: pointer; padding: 0.75rem; border-radius: 0.5rem; transition: background 0.2s ease;" 
                               onmouseover="this.style.background='rgba(220, 38, 38, 0.15)'" 
                               onmouseout="this.style.background='transparent'">
                            <input type="checkbox" id="disable_vote_removal" name="disable_vote_removal" 
                                   style="width: 22px; height: 22px; cursor: pointer; accent-color: #dc2626; border-radius: 4px; margin-top: 2px; flex-shrink: 0;">
                            <div style="flex: 1;">
                                <span style="color: rgba(255, 255, 255, 0.85); font-weight: 500; user-select: none; display: block; margin-bottom: 0.25rem;">
                                    غیرفعال کردن امکان پس گرفتن رای
                                </span>
                                <span style="color: rgba(255, 255, 255, 0.6); font-size: 0.875rem; line-height: 1.5; display: block;">
                                    اگر این گزینه فعال باشد، کاربران پس از رای دادن نمی‌توانند رای خود را پس بگیرند
                                </span>
                            </div>
                        </label>
                    </div>
                </div>
                
                <div style="display: flex; gap: 1rem; margin-top: 1.5rem;">
                    <button type="submit" class="btn btn-primary">ذخیره</button>
                    <button type="button" class="btn btn-secondary" onclick="closeSuggestionModal()">انصراف</button>
                </div>
            </form>
        </div>
    </div>

    <script>
        function resetForm() {
            document.getElementById('suggestionForm').reset();
            document.getElementById('formAction').value = 'create';
            document.getElementById('suggestionId').value = '';
            document.getElementById('modalTitle').textContent = 'ایجاد پیشنهاد جدید';
            document.getElementById('is_active').checked = true;
            document.getElementById('is_featured').checked = false;
            document.getElementById('disable_vote_removal').checked = false;
        }

        function showSuggestionModal() {
            document.getElementById('suggestionModal').style.display = 'block';
        }

        function closeSuggestionModal() {
            document.getElementById('suggestionModal').style.display = 'none';
        }

        function editSuggestion(suggestion) {
            document.getElementById('formAction').value = 'update';
            document.getElementById('suggestionId').value = suggestion.id;
            document.getElementById('title').value = suggestion.title;
            document.getElementById('description').value = suggestion.description;
            document.getElementById('display_order').value = suggestion.display_order;
            document.getElementById('is_active').checked = suggestion.is_active == 1;
            document.getElementById('is_featured').checked = suggestion.is_featured == 1;
            document.getElementById('disable_vote_removal').checked = suggestion.disable_vote_removal == 1;
            document.getElementById('modalTitle').textContent = 'ویرایش پیشنهاد';
            
            showSuggestionModal();
        }

        // Close modal when clicking outside
        window.onclick = function(event) {
            const modal = document.getElementById('suggestionModal');
            if (event.target === modal) {
                closeSuggestionModal();
            }
        }

        // Close modal when pressing Escape key
        document.addEventListener('keydown', function(event) {
            if (event.key === 'Escape') {
                closeSuggestionModal();
            }
        });
    </script>
</body>
</html>
