<?php
// Set proper charset headers
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');
mb_http_output('UTF-8');

// Start session
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once 'config/database.php';
require_once 'includes/functions.php';

// Check authentication
if (!isLoggedIn()) {
    logSecurityEvent('unauthorized_access_attempt', 'Unauthorized access attempt to bug-reports.php from IP: ' . getClientIP(), 'high');
    header('Location: 403.php');
    exit;
}
checkSessionTimeout();

$message = '';
$error = '';

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        switch ($action) {
            case 'update_status':
                $bugId = (int)($_POST['bug_id'] ?? 0);
                $status = sanitizeInput($_POST['status'] ?? '');
                $priority = sanitizeInput($_POST['priority'] ?? '');
                
                if ($bugId && $status) {
                    $updateQuery = "UPDATE bug_reports SET status = ?, priority = ?, updated_at = NOW() WHERE id = ?";
                    $updateStmt = $conn->prepare($updateQuery);
                    $updateStmt->execute([$status, $priority, $bugId]);
                    
                    $message = "وضعیت گزارش باگ با موفقیت بروزرسانی شد";
                    logSecurityEvent('bug_status_updated', 'Bug ID: ' . $bugId . ', Status: ' . $status, 'low');
                }
                break;
                
            case 'bulk_update':
                $selectedBugs = $_POST['selected_bugs'] ?? [];
                $bulkStatus = sanitizeInput($_POST['bulk_status'] ?? '');
                $bulkPriority = sanitizeInput($_POST['bulk_priority'] ?? '');
                
                if (!empty($selectedBugs) && $bulkStatus) {
                    $updateQuery = "UPDATE bug_reports SET status = ?, priority = ?, updated_at = NOW() WHERE id IN (" . str_repeat('?,', count($selectedBugs) - 1) . "?)";
                    $params = array_merge([$bulkStatus, $bulkPriority], $selectedBugs);
                    $updateStmt = $conn->prepare($updateQuery);
                    $updateStmt->execute($params);
                    
                    $message = count($selectedBugs) . " گزارش باگ با موفقیت بروزرسانی شد";
                    logSecurityEvent('bugs_bulk_updated', 'Count: ' . count($selectedBugs), 'low');
                }
                break;
                
            case 'delete':
                $bugId = (int)($_POST['bug_id'] ?? 0);
                
                if ($bugId) {
                    $deleteQuery = "DELETE FROM bug_reports WHERE id = ?";
                    $deleteStmt = $conn->prepare($deleteQuery);
                    $deleteStmt->execute([$bugId]);
                    
                    $message = "گزارش باگ با موفقیت حذف شد";
                    logSecurityEvent('bug_deleted', 'Bug ID: ' . $bugId, 'medium');
                }
                break;
                
            case 'bulk_delete':
                $selectedBugs = $_POST['selected_bugs'] ?? [];
                
                if (!empty($selectedBugs)) {
                    $deleteQuery = "DELETE FROM bug_reports WHERE id IN (" . str_repeat('?,', count($selectedBugs) - 1) . "?)";
                    $deleteStmt = $conn->prepare($deleteQuery);
                    $deleteStmt->execute($selectedBugs);
                    
                    $message = count($selectedBugs) . " گزارش باگ با موفقیت حذف شد";
                    logSecurityEvent('bugs_bulk_deleted', 'Count: ' . count($selectedBugs), 'medium');
                }
                break;
                
            case 'get_bug_details':
                $bugId = (int)($_POST['bug_id'] ?? 0);
                
                if ($bugId) {
                    // Check if attachment_path column exists
                    try {
                        $checkColumnQuery = "SHOW COLUMNS FROM bug_reports LIKE 'attachment_path'";
                        $checkColumnStmt = $conn->query($checkColumnQuery);
                        $hasAttachmentColumn = $checkColumnStmt->rowCount() > 0;
                        $checkColumnStmt->closeCursor();
                    } catch (Exception $e) {
                        $hasAttachmentColumn = false;
                    }
                    
                    // Build query - ALWAYS try to select attachment_path, even if column check fails
                    // Use COALESCE to handle NULL values gracefully
                    $bugQuery = "SELECT id, name, email, subject, description, priority, status, ip_address, user_agent, 
                                COALESCE(attachment_path, '') as attachment_path, 
                                admin_response, response_sent_at, created_at, updated_at 
                                FROM bug_reports WHERE id = ?";
                    
                    $bugStmt = $conn->prepare($bugQuery);
                    $bugStmt->execute([$bugId]);
                    $bug = $bugStmt->fetch(PDO::FETCH_ASSOC);
                    $bugStmt->closeCursor();
                    
                    if ($bug) {
                        header('Content-Type: application/json');
                        
                        // Process attachment_path - check if it exists in the result
                        $attachmentPath = null;
                        // Check if attachment_path column exists in the result (even if it's empty string from COALESCE)
                        if (isset($bug['attachment_path'])) {
                            $rawAttachmentPath = $bug['attachment_path'];
                            // Check if it's not null, not empty string, and not the string 'null'
                            if ($rawAttachmentPath !== null && $rawAttachmentPath !== '' && trim($rawAttachmentPath) !== '' && strtolower(trim($rawAttachmentPath)) !== 'null') {
                                $attachmentPath = trim($rawAttachmentPath);
                            }
                        }
                        
                        // Also try direct query to verify
                        try {
                            $directQuery = "SELECT attachment_path FROM bug_reports WHERE id = ?";
                            $directStmt = $conn->prepare($directQuery);
                            $directStmt->execute([$bugId]);
                            $directResult = $directStmt->fetch(PDO::FETCH_ASSOC);
                            $directStmt->closeCursor();
                            if ($directResult && !empty($directResult['attachment_path'])) {
                                $attachmentPath = trim($directResult['attachment_path']);
                                error_log("Direct query found attachment_path: " . var_export($attachmentPath, true));
                            }
                        } catch (Exception $e) {
                            error_log("Direct query error: " . $e->getMessage());
                        }
                        
                        $bugData = [
                            'id' => $bug['id'],
                            'subject' => htmlspecialchars($bug['subject'], ENT_QUOTES, 'UTF-8'),
                            'description' => htmlspecialchars($bug['description'], ENT_QUOTES, 'UTF-8'),
                            'name' => htmlspecialchars($bug['name'], ENT_QUOTES, 'UTF-8'),
                            'email' => htmlspecialchars($bug['email'], ENT_QUOTES, 'UTF-8'),
                            'status' => $bug['status'],
                            'priority' => $bug['priority'],
                            'created_at' => $bug['created_at'],
                            'attachment_path' => $attachmentPath // Always include, even if null
                        ];
                        
                        // Debug logging - show all bug data keys and raw values
                        error_log("get_bug_details - Bug ID: $bugId | hasAttachmentColumn: " . ($hasAttachmentColumn ? 'yes' : 'no') . " | attachment_path from DB (raw): " . var_export($bug['attachment_path'] ?? 'not set', true) . " | attachment_path (processed): " . var_export($attachmentPath, true) . " | All bug keys: " . implode(', ', array_keys($bug)));
                        
                        echo json_encode([
                            'success' => true,
                            'bug' => $bugData
                        ], JSON_UNESCAPED_UNICODE);
                    } else {
                        header('Content-Type: application/json');
                        echo json_encode(['success' => false, 'message' => 'گزارش باگ یافت نشد']);
                    }
                } else {
                    header('Content-Type: application/json');
                    echo json_encode(['success' => false, 'message' => 'شناسه گزارش نامعتبر است']);
                }
                exit;
                break;
        }
    } catch (Exception $e) {
        error_log("Bug reports error: " . $e->getMessage());
        $error = 'خطا در عملیات. لطفاً دوباره تلاش کنید';
    }
}

// Get bug reports with filtering
$statusFilter = $_GET['status'] ?? '';
$priorityFilter = $_GET['priority'] ?? '';
$searchQuery = $_GET['search'] ?? '';

try {
    $db = new Database();
    $conn = $db->getConnection();
    
    $whereConditions = [];
    $params = [];
    
    if ($statusFilter) {
        $whereConditions[] = "status = ?";
        $params[] = $statusFilter;
    }
    
    if ($priorityFilter) {
        $whereConditions[] = "priority = ?";
        $params[] = $priorityFilter;
    }
    
    if ($searchQuery) {
        $whereConditions[] = "(subject LIKE ? OR description LIKE ? OR name LIKE ? OR email LIKE ?)";
        $searchParam = '%' . $searchQuery . '%';
        $params[] = $searchParam;
        $params[] = $searchParam;
        $params[] = $searchParam;
        $params[] = $searchParam;
    }
    
    $whereClause = '';
    if (!empty($whereConditions)) {
        $whereClause = 'WHERE ' . implode(' AND ', $whereConditions);
    }
    
    // Check if new columns exist, otherwise use basic query
    try {
        $checkColumnsQuery = "SHOW COLUMNS FROM bug_reports LIKE 'attachment_path'";
        $checkStmt = $conn->query($checkColumnsQuery);
        $hasNewColumns = $checkStmt->rowCount() > 0;
        $checkStmt->closeCursor();
    } catch (Exception $e) {
        $hasNewColumns = false;
    }
    
    if ($hasNewColumns) {
        $bugsQuery = "SELECT id, name, email, subject, description, priority, status, ip_address, user_agent, attachment_path, admin_response, response_sent_at, created_at, updated_at FROM bug_reports $whereClause ORDER BY created_at DESC";
    } else {
        $bugsQuery = "SELECT id, name, email, subject, description, priority, status, ip_address, user_agent, created_at, updated_at FROM bug_reports $whereClause ORDER BY created_at DESC";
    }
    
    $bugsStmt = $conn->prepare($bugsQuery);
    $bugsStmt->execute($params);
    $bugs = $bugsStmt->fetchAll(PDO::FETCH_ASSOC);
    $bugsStmt->closeCursor();
    
    // Add default values for new columns if they don't exist
    if (!$hasNewColumns) {
        foreach ($bugs as &$bug) {
            $bug['attachment_path'] = null;
            $bug['admin_response'] = null;
            $bug['response_sent_at'] = null;
        }
        unset($bug);
    } else {
        // Debug: Log attachment_path for first bug
        if (!empty($bugs) && isset($bugs[0]['attachment_path'])) {
            error_log("First bug attachment_path: '" . $bugs[0]['attachment_path'] . "' | Type: " . gettype($bugs[0]['attachment_path']));
        }
    }
    
    // Get statistics
    $statsQuery = "SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN status = 'new' THEN 1 ELSE 0 END) as new_count,
        SUM(CASE WHEN status = 'in_progress' THEN 1 ELSE 0 END) as in_progress_count,
        SUM(CASE WHEN status = 'resolved' THEN 1 ELSE 0 END) as resolved_count,
        SUM(CASE WHEN status = 'closed' THEN 1 ELSE 0 END) as closed_count
        FROM bug_reports";
    $statsStmt = $conn->query($statsQuery);
    $stats = $statsStmt->fetch();
    
} catch (Exception $e) {
    error_log("Bug reports list error: " . $e->getMessage());
    $error = 'خطا در بارگذاری گزارش‌های باگ';
    $bugs = [];
    $stats = ['total' => 0, 'new_count' => 0, 'in_progress_count' => 0, 'resolved_count' => 0, 'closed_count' => 0];
}
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>گزارش‌های باگ - پنل مدیریت گواهیتو</title>
    <link rel="stylesheet" href="assets/admin-style.css">
    <link rel="icon" type="image/png" href="assets/images/logo.png">
    <?php include 'includes/admin-styles.php'; ?>
    <style>
        /* Notification System */
        .notification-container {
            position: fixed;
            top: 20px;
            left: 20px;
            z-index: 10000;
            display: flex;
            flex-direction: column;
            gap: 10px;
        }
        
        .notification {
            min-width: 300px;
            max-width: 500px;
            padding: 1rem 1.5rem;
            border-radius: 12px;
            box-shadow: 0 8px 24px rgba(0, 0, 0, 0.3);
            display: flex;
            align-items: center;
            gap: 1rem;
            animation: slideInLeft 0.3s ease-out;
            font-family: 'Vazirmatn', 'Tahoma', Arial, sans-serif;
        }
        
        .notification-success {
            background: linear-gradient(135deg, #27ae60, #2ecc71);
            color: white;
            border: 2px solid rgba(255, 255, 255, 0.2);
        }
        
        .notification-error {
            background: linear-gradient(135deg, #e74c3c, #c0392b);
            color: white;
            border: 2px solid rgba(255, 255, 255, 0.2);
        }
        
        .notification-warning {
            background: linear-gradient(135deg, #f39c12, #e67e22);
            color: white;
            border: 2px solid rgba(255, 255, 255, 0.2);
        }
        
        .notification-icon {
            font-size: 1.5rem;
            flex-shrink: 0;
        }
        
        .notification-content {
            flex: 1;
        }
        
        .notification-title {
            font-weight: 700;
            font-size: 1rem;
            margin-bottom: 0.25rem;
        }
        
        .notification-message {
            font-size: 0.875rem;
            opacity: 0.95;
            line-height: 1.5;
        }
        
        .notification-close {
            background: rgba(255, 255, 255, 0.2);
            border: none;
            color: white;
            width: 24px;
            height: 24px;
            border-radius: 50%;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.2rem;
            flex-shrink: 0;
            transition: background 0.2s;
        }
        
        .notification-close:hover {
            background: rgba(255, 255, 255, 0.3);
        }
        
        @keyframes slideInLeft {
            from {
                transform: translateX(-100%);
                opacity: 0;
            }
            to {
                transform: translateX(0);
                opacity: 1;
            }
        }
        
        @keyframes slideOutLeft {
            from {
                transform: translateX(0);
                opacity: 1;
            }
            to {
                transform: translateX(-100%);
                opacity: 0;
            }
        }
        
        .notification.hiding {
            animation: slideOutLeft 0.3s ease-out forwards;
        }
    </style>
</head>
<body>
    <!-- Notification Container -->
    <div id="notificationContainer" class="notification-container"></div>
    <?php include 'includes/header.php'; ?>
    <?php include 'includes/sidebar.php'; ?>
    
    <!-- Sidebar Overlay for Mobile -->
    <div class="sidebar-overlay" id="sidebarOverlay"></div>

    <!-- Main Content -->
    <div class="main-content" id="mainContent">
        <div class="page-header">
            <h1 class="page-title">🐛 گزارش‌های باگ</h1>
            <p class="page-description">مدیریت و پیگیری گزارش‌های باگ سیستم</p>
        </div>

        <?php if ($error): ?>
            <div class="alert alert-danger">
                <?php echo htmlspecialchars($error); ?>
            </div>
        <?php endif; ?>

        <?php if ($message): ?>
            <div class="alert alert-success">
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>

        <!-- Statistics -->
        <div class="grid grid-cols-1 grid-cols-2 grid-cols-3 grid-cols-4" style="margin-bottom: 2rem;">
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['total']; ?></div>
                <div class="stats-label">کل گزارش‌ها</div>
            </div>
            
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['new_count']; ?></div>
                <div class="stats-label">جدید</div>
            </div>
            
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['in_progress_count']; ?></div>
                <div class="stats-label">در حال بررسی</div>
            </div>
            
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['resolved_count']; ?></div>
                <div class="stats-label">حل شده</div>
            </div>
            
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['closed_count']; ?></div>
                <div class="stats-label">بسته</div>
            </div>
        </div>

        <!-- Filters -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">🔍 فیلترها</h2>
            </div>
            
            <form method="GET" action="" style="display: flex; gap: 1rem; align-items: end; flex-wrap: wrap;">
                <div class="form-group" style="flex: 1; min-width: 200px;">
                    <label for="search" class="form-label">جستجو</label>
                    <input type="text" id="search" name="search" class="form-input" 
                           value="<?php echo htmlspecialchars($searchQuery); ?>" 
                           placeholder="جستجو در موضوع، توضیحات، نام یا ایمیل">
                </div>
                
                <div class="form-group" style="min-width: 150px;">
                    <label for="status" class="form-label">وضعیت</label>
                    <select id="status" name="status" class="form-select">
                        <option value="">همه</option>
                        <option value="new" <?php echo $statusFilter === 'new' ? 'selected' : ''; ?>>جدید</option>
                        <option value="in_progress" <?php echo $statusFilter === 'in_progress' ? 'selected' : ''; ?>>در حال بررسی</option>
                        <option value="resolved" <?php echo $statusFilter === 'resolved' ? 'selected' : ''; ?>>حل شده</option>
                        <option value="closed" <?php echo $statusFilter === 'closed' ? 'selected' : ''; ?>>بسته</option>
                    </select>
                </div>
                
                <div class="form-group" style="min-width: 150px;">
                    <label for="priority" class="form-label">اولویت</label>
                    <select id="priority" name="priority" class="form-select">
                        <option value="">همه</option>
                        <option value="low" <?php echo $priorityFilter === 'low' ? 'selected' : ''; ?>>کم</option>
                        <option value="medium" <?php echo $priorityFilter === 'medium' ? 'selected' : ''; ?>>متوسط</option>
                        <option value="high" <?php echo $priorityFilter === 'high' ? 'selected' : ''; ?>>زیاد</option>
                        <option value="critical" <?php echo $priorityFilter === 'critical' ? 'selected' : ''; ?>>بحرانی</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <button type="submit" class="btn btn-primary">اعمال فیلتر</button>
                    <a href="bug-reports.php" class="btn btn-secondary">پاک کردن</a>
                </div>
            </form>
        </div>

        <!-- Bug Reports List -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">📋 لیست گزارش‌های باگ</h2>
                <div style="display: flex; gap: 0.5rem;">
                    <button type="button" class="btn btn-warning btn-sm" onclick="showBulkUpdateModal()">
                        بروزرسانی دسته‌ای
                    </button>
                    <button type="button" class="btn btn-danger btn-sm" onclick="deleteSelectedBugs()">
                        حذف انتخاب شده‌ها
                    </button>
                </div>
            </div>
            
            <?php if (!empty($bugs)): ?>
                <form method="POST" action="" id="bugsForm">
                    <input type="hidden" name="action" value="bulk_delete">
                    
                    <div class="table-container">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>
                                        <input type="checkbox" id="selectAll" onchange="toggleSelectAll()">
                                    </th>
                                    <th>موضوع</th>
                                    <th>گزارش‌دهنده</th>
                                    <th>اولویت</th>
                                    <th>وضعیت</th>
                                    <th>تاریخ</th>
                                    <th>عملیات</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($bugs as $bug): ?>
                                    <tr>
                                        <td>
                                            <input type="checkbox" name="selected_bugs[]" value="<?php echo $bug['id']; ?>" class="bug-checkbox">
                                        </td>
                                        <td>
                                            <strong><?php echo htmlspecialchars($bug['subject']); ?></strong>
                                            <br>
                                            <small style="color: var(--dark-400);">
                                                <?php echo htmlspecialchars(substr($bug['description'], 0, 100)) . (strlen($bug['description']) > 100 ? '...' : ''); ?>
                                            </small>
                                            <?php 
                                            $attachmentPath = isset($bug['attachment_path']) ? trim($bug['attachment_path']) : '';
                                            if (!empty($attachmentPath) && $attachmentPath !== 'null'): ?>
                                                <br>
                                                <small style="color: var(--primary-400); margin-top: 0.25rem; display: inline-block;">
                                                    📎 <a href="https://gavahito.ir/backend/<?php echo htmlspecialchars($attachmentPath); ?>" target="_blank" style="color: var(--primary-400); text-decoration: none;">مشاهده تصویر</a>
                                                    | 
                                                    <a href="https://gavahito.ir/backend/<?php echo htmlspecialchars($attachmentPath); ?>" download style="color: var(--secondary); text-decoration: none;">💾 دانلود</a>
                                                </small>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <div><?php echo htmlspecialchars($bug['name']); ?></div>
                                            <small style="color: var(--dark-400);"><?php echo htmlspecialchars($bug['email']); ?></small>
                                        </td>
                                        <td><?php echo getPriorityBadge($bug['priority']); ?></td>
                                        <td><?php echo getStatusBadge($bug['status']); ?></td>
                                        <td><?php echo formatDate($bug['created_at']); ?></td>
                                        <td>
                                            <div style="display: flex; gap: 0.5rem; flex-wrap: wrap;">
                                                <button type="button" class="btn btn-warning btn-sm" onclick="editBug(<?php echo $bug['id']; ?>, '<?php echo htmlspecialchars($bug['subject']); ?>', '<?php echo $bug['status']; ?>', '<?php echo $bug['priority']; ?>')">
                                                    ویرایش
                                                </button>
                                                <button type="button" class="btn btn-info btn-sm" onclick="viewBugAjax(<?php echo $bug['id']; ?>)">
                                                    مشاهده
                                                </button>
                                                <button type="button" class="btn btn-primary btn-sm" onclick="showResponseModal(<?php echo $bug['id']; ?>, '<?php echo htmlspecialchars($bug['subject']); ?>', '<?php echo htmlspecialchars($bug['email']); ?>', '<?php echo htmlspecialchars($bug['name']); ?>', <?php echo $bug['admin_response'] ? 'true' : 'false'; ?>, '<?php echo htmlspecialchars($bug['admin_response'] ?? ''); ?>')">
                                                    <?php echo $bug['admin_response'] ? '✏️ ویرایش پاسخ' : '💬 پاسخ'; ?>
                                                </button>
                                                <button type="button" class="btn btn-danger btn-sm" onclick="deleteBug(<?php echo $bug['id']; ?>)">
                                                    حذف
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </form>
            <?php else: ?>
                <p style="color: var(--dark-400); text-align: center; padding: 2rem;">هیچ گزارش باگی یافت نشد</p>
            <?php endif; ?>
        </div>
    </div>

    <!-- Edit Bug Modal -->
    <div id="editModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 1000;">
        <div style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); background: var(--dark-800); padding: 2rem; border-radius: 1rem; min-width: 400px;">
            <h3 style="margin-bottom: 1rem; color: var(--primary-400);">ویرایش گزارش باگ</h3>
            
            <form method="POST" action="" id="editBugForm">
                <input type="hidden" name="action" value="update_status">
                <input type="hidden" name="bug_id" id="editBugId">
                
                <div class="form-group">
                    <label for="editSubject" class="form-label">موضوع</label>
                    <input type="text" id="editSubject" class="form-input" readonly>
                </div>
                
                <div class="form-group">
                    <label for="editStatus" class="form-label">وضعیت</label>
                    <select id="editStatus" name="status" class="form-select">
                        <option value="new">جدید</option>
                        <option value="in_progress">در حال بررسی</option>
                        <option value="resolved">حل شده</option>
                        <option value="closed">بسته</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="editPriority" class="form-label">اولویت</label>
                    <select id="editPriority" name="priority" class="form-select">
                        <option value="low">کم</option>
                        <option value="medium">متوسط</option>
                        <option value="high">زیاد</option>
                        <option value="critical">بحرانی</option>
                    </select>
                </div>
                
                <div style="display: flex; gap: 1rem; margin-top: 1rem;">
                    <button type="submit" class="btn btn-primary">ذخیره تغییرات</button>
                    <button type="button" class="btn btn-secondary" onclick="closeEditModal()">انصراف</button>
                </div>
            </form>
        </div>
    </div>

    <!-- View Bug Modal -->
    <div id="viewModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 1000;">
        <div style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); background: var(--dark-800); padding: 2rem; border-radius: 1rem; min-width: 500px; max-width: 90%; max-height: 90%; display: flex; flex-direction: column; box-shadow: 0 20px 40px rgba(0,0,0,0.3);">
            <h3 style="margin-bottom: 1rem; color: var(--primary-400); flex-shrink: 0;">مشاهده گزارش باگ</h3>
            
            <div id="viewBugContent" style="flex: 1; overflow-y: auto; padding-right: 0.5rem; scrollbar-width: thin; scrollbar-color: var(--dark-600) var(--dark-800);">
                <!-- Content will be filled by JavaScript -->
            </div>
            
            <div style="display: flex; gap: 1rem; margin-top: 1rem; flex-shrink: 0;">
                <button type="button" class="btn btn-secondary" onclick="closeViewModal()">بستن</button>
            </div>
        </div>
    </div>

    <!-- Response Modal -->
    <div id="responseModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 1000;">
        <div style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); background: var(--dark-800); padding: 2rem; border-radius: 1rem; min-width: 500px; max-width: 90%; max-height: 90%; display: flex; flex-direction: column; box-shadow: 0 20px 40px rgba(0,0,0,0.3);">
            <h3 style="margin-bottom: 1rem; color: var(--primary-400); flex-shrink: 0;">💬 ارسال پاسخ</h3>
            
            <div id="responseBugInfo" style="margin-bottom: 1rem; padding: 1rem; background: var(--dark-700); border-radius: 0.5rem; flex-shrink: 0;">
                <!-- Bug info will be filled by JavaScript -->
            </div>
            
            <form id="responseForm" style="flex: 1; display: flex; flex-direction: column;">
                <input type="hidden" id="responseBugId" value="">
                <input type="hidden" id="responseFormType" value="bug_report">
                
                <div class="form-group" style="flex: 1; display: flex; flex-direction: column;">
                    <label for="responseText" class="form-label">متن پاسخ *</label>
                    <textarea id="responseText" class="form-input" rows="10" required style="flex: 1; resize: none;" placeholder="پاسخ خود را اینجا بنویسید..."></textarea>
                </div>
                
                <div id="existingResponse" style="margin-top: 1rem; padding: 1rem; background: rgba(197, 145, 44, 0.1); border: 1px solid rgba(197, 145, 44, 0.3); border-radius: 0.5rem; display: none;">
                    <p style="color: var(--secondary-color); font-weight: 600; margin-bottom: 0.5rem;">📝 پاسخ قبلی:</p>
                    <p id="existingResponseText" style="color: var(--dark-300); white-space: pre-wrap;"></p>
                </div>
                
                <div style="display: flex; gap: 1rem; margin-top: 1rem; flex-shrink: 0;">
                    <button type="submit" class="btn btn-primary" id="responseSubmitBtn">📧 ارسال پاسخ</button>
                    <button type="button" class="btn btn-secondary" onclick="closeResponseModal()">انصراف</button>
                </div>
            </form>
        </div>
    </div>

    <!-- Bulk Update Modal -->
    <div id="bulkUpdateModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 1000;">
        <div style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); background: var(--dark-800); padding: 2rem; border-radius: 1rem; min-width: 400px;">
            <h3 style="margin-bottom: 1rem; color: var(--primary-400);">بروزرسانی دسته‌ای</h3>
            <p id="bulkUpdateCount" style="margin-bottom: 1rem; color: var(--dark-300);"></p>
            
            <form method="POST" action="" id="bulkUpdateForm">
                <input type="hidden" name="action" value="bulk_update">
                <div id="bulkUpdateSelectedBugs">
                    <!-- Selected bug IDs will be added here dynamically -->
                </div>
                
                <div class="form-group">
                    <label for="bulkStatus" class="form-label">وضعیت جدید</label>
                    <select id="bulkStatus" name="bulk_status" class="form-select">
                        <option value="">انتخاب کنید</option>
                        <option value="new">جدید</option>
                        <option value="in_progress">در حال بررسی</option>
                        <option value="resolved">حل شده</option>
                        <option value="closed">بسته</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="bulkPriority" class="form-label">اولویت جدید</label>
                    <select id="bulkPriority" name="bulk_priority" class="form-select">
                        <option value="">انتخاب کنید</option>
                        <option value="low">کم</option>
                        <option value="medium">متوسط</option>
                        <option value="high">زیاد</option>
                        <option value="critical">بحرانی</option>
                    </select>
                </div>
                
                <div style="display: flex; gap: 1rem; margin-top: 1rem;">
                    <button type="submit" class="btn btn-primary">بروزرسانی</button>
                    <button type="button" class="btn btn-secondary" onclick="closeBulkUpdateModal()">انصراف</button>
                </div>
            </form>
        </div>
    </div>

    <script>
        // Toggle select all checkbox
        function toggleSelectAll() {
            const selectAll = document.getElementById('selectAll');
            const checkboxes = document.querySelectorAll('.bug-checkbox');
            
            checkboxes.forEach(checkbox => {
                checkbox.checked = selectAll.checked;
            });
        }
        
        // Delete selected bugs
        function deleteSelectedBugs() {
            const checkboxes = document.querySelectorAll('.bug-checkbox:checked');
            
            if (checkboxes.length === 0) {
                alert('لطفاً حداقل یک گزارش را انتخاب کنید');
                return;
            }
            
            if (confirm('آیا از حذف ' + checkboxes.length + ' گزارش انتخاب شده اطمینان دارید؟')) {
                document.getElementById('bugsForm').submit();
            }
        }
        
        // Edit bug modal
        function editBug(bugId, subject, status, priority) {
            document.getElementById('editBugId').value = bugId;
            document.getElementById('editSubject').value = subject;
            document.getElementById('editStatus').value = status;
            document.getElementById('editPriority').value = priority;
            document.getElementById('editModal').style.display = 'block';
        }
        
        function closeEditModal() {
            document.getElementById('editModal').style.display = 'none';
        }
        
        // View bug modal using AJAX
        function viewBugAjax(bugId) {
            // Show loading state
            document.getElementById('viewBugContent').innerHTML = '<div style="text-align: center; padding: 2rem; color: var(--dark-300);">در حال بارگذاری...</div>';
            document.getElementById('viewModal').style.display = 'block';
            
            // Create form data
            const formData = new FormData();
            formData.append('action', 'get_bug_details');
            formData.append('bug_id', bugId);
            
            // Send AJAX request
            fetch('bug-reports.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.text())
            .then(data => {
                try {
                    const bugData = JSON.parse(data);
                    console.log('Bug data received:', bugData); // Debug log
                    if (bugData.success) {
                        console.log('Attachment path:', bugData.bug.attachment_path); // Debug log
                        console.log('Attachment path type:', typeof bugData.bug.attachment_path); // Debug log
                        console.log('Attachment path empty check:', !bugData.bug.attachment_path); // Debug log
                        console.log('Attachment path null check:', bugData.bug.attachment_path === null); // Debug log
                        console.log('Attachment path string check:', bugData.bug.attachment_path === ''); // Debug log
                        let content = `
                            <div class="form-group">
                                <label class="form-label">موضوع</label>
                                <div style="padding: 0.75rem; background: var(--dark-700); border-radius: 0.5rem; border: 1px solid var(--dark-600); word-wrap: break-word; overflow-wrap: break-word;">
                                    ${bugData.bug.subject}
                                </div>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label">توضیحات</label>
                                <div style="padding: 0.75rem; background: var(--dark-700); border-radius: 0.5rem; border: 1px solid var(--dark-600); white-space: pre-wrap; word-wrap: break-word; overflow-wrap: break-word; line-height: 1.6; max-height: none; min-height: 100px;">
                                    ${bugData.bug.description}
                                </div>
                            </div>
                            
                            ${bugData.bug.attachment_path && bugData.bug.attachment_path !== null && bugData.bug.attachment_path !== '' ? `
                            <div class="form-group">
                                <label class="form-label">📷 ضمیمه تصویر</label>
                                <div style="padding: 1rem; background: var(--dark-700); border-radius: 0.5rem; border: 1px solid var(--dark-600); text-align: center;">
                                    <img src="https://gavahito.ir/backend/${bugData.bug.attachment_path}" alt="ضمیمه" style="max-width: 100%; max-height: 500px; border-radius: 0.5rem; cursor: pointer; display: block; margin: 0 auto; border: 2px solid var(--primary-400);" onclick="window.open('https://gavahito.ir/backend/${bugData.bug.attachment_path}', '_blank')" onerror="console.error('Image load error:', this.src); this.style.display='none'; this.nextElementSibling.style.display='block';">
                                    <div style="display: none; text-align: center; padding: 1rem; color: var(--danger);">
                                        <p>❌ خطا در بارگذاری تصویر</p>
                                        <a href="https://gavahito.ir/backend/${bugData.bug.attachment_path}" target="_blank" style="color: var(--primary-400); text-decoration: none;">🔗 تلاش برای باز کردن مستقیم</a>
                                    </div>
                                    <br>
                                    <div style="display: flex; gap: 0.5rem; justify-content: center; flex-wrap: wrap; margin-top: 1rem;">
                                        <a href="https://gavahito.ir/backend/${bugData.bug.attachment_path}" target="_blank" style="color: white; text-decoration: none; padding: 0.5rem 1rem; background: var(--primary-400); border-radius: 0.5rem; display: inline-block; font-weight: 600;">🔗 باز کردن در تب جدید</a>
                                        <a href="https://gavahito.ir/backend/${bugData.bug.attachment_path}" download style="color: white; text-decoration: none; padding: 0.5rem 1rem; background: var(--secondary); border-radius: 0.5rem; display: inline-block; font-weight: 600;">💾 دانلود تصویر</a>
                                    </div>
                                </div>
                            </div>
                            ` : '<div class="form-group"><label class="form-label">📷 ضمیمه تصویر</label><div style="padding: 0.75rem; background: var(--dark-700); border-radius: 0.5rem; border: 1px solid var(--dark-600); color: var(--dark-400); text-align: center;">هیچ تصویری ضمیمه نشده است</div></div>'}
                            
                            <div class="form-group">
                                <label class="form-label">گزارش‌دهنده</label>
                                <div style="padding: 0.75rem; background: var(--dark-700); border-radius: 0.5rem; border: 1px solid var(--dark-600); word-wrap: break-word; overflow-wrap: break-word;">
                                    ${bugData.bug.name} (${bugData.bug.email})
                                </div>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label">شناسه گزارش</label>
                                <div style="padding: 0.75rem; background: var(--dark-700); border-radius: 0.5rem; border: 1px solid var(--dark-600);">
                                    #${bugData.bug.id}
                                </div>
                            </div>
                        `;
                        document.getElementById('viewBugContent').innerHTML = content;
                    } else {
                        document.getElementById('viewBugContent').innerHTML = '<div style="text-align: center; padding: 2rem; color: var(--dark-300);">خطا در بارگذاری اطلاعات</div>';
                    }
                } catch (e) {
                    document.getElementById('viewBugContent').innerHTML = '<div style="text-align: center; padding: 2rem; color: var(--dark-300);">خطا در پردازش اطلاعات</div>';
                }
            })
            .catch(error => {
                document.getElementById('viewBugContent').innerHTML = '<div style="text-align: center; padding: 2rem; color: var(--dark-300);">خطا در ارتباط با سرور</div>';
            });
        }
        
        // Legacy view bug modal (for backward compatibility)
        function viewBug(bugId, subject, description, name, email) {
            // This function is kept for backward compatibility
            const content = `
                <div class="form-group">
                    <label class="form-label">موضوع</label>
                    <div style="padding: 0.75rem; background: var(--dark-700); border-radius: 0.5rem; border: 1px solid var(--dark-600); word-wrap: break-word; overflow-wrap: break-word;">
                        ${subject}
                    </div>
                </div>
                
                <div class="form-group">
                    <label class="form-label">توضیحات</label>
                    <div style="padding: 0.75rem; background: var(--dark-700); border-radius: 0.5rem; border: 1px solid var(--dark-600); white-space: pre-wrap; word-wrap: break-word; overflow-wrap: break-word; line-height: 1.6; max-height: none; min-height: 100px;">
                        ${description}
                    </div>
                </div>
                
                <div class="form-group">
                    <label class="form-label">گزارش‌دهنده</label>
                    <div style="padding: 0.75rem; background: var(--dark-700); border-radius: 0.5rem; border: 1px solid var(--dark-600); word-wrap: break-word; overflow-wrap: break-word;">
                        ${name} (${email})
                    </div>
                </div>
                
                <div class="form-group">
                    <label class="form-label">شناسه گزارش</label>
                    <div style="padding: 0.75rem; background: var(--dark-700); border-radius: 0.5rem; border: 1px solid var(--dark-600);">
                        #${bugId}
                    </div>
                </div>
            `;
            
            document.getElementById('viewBugContent').innerHTML = content;
            document.getElementById('viewModal').style.display = 'block';
        }
        
        function closeViewModal() {
            document.getElementById('viewModal').style.display = 'none';
        }
        
        // Bulk update modal
        function showBulkUpdateModal() {
            const checkboxes = document.querySelectorAll('.bug-checkbox:checked');
            
            if (checkboxes.length === 0) {
                alert('لطفاً حداقل یک گزارش را انتخاب کنید');
                return;
            }
            
            // Clear previous form data
            document.getElementById('bulkStatus').value = '';
            document.getElementById('bulkPriority').value = '';
            
            // Add selected bug IDs to the form
            const selectedBugsContainer = document.getElementById('bulkUpdateSelectedBugs');
            selectedBugsContainer.innerHTML = '';
            
            checkboxes.forEach(checkbox => {
                const hiddenInput = document.createElement('input');
                hiddenInput.type = 'hidden';
                hiddenInput.name = 'selected_bugs[]';
                hiddenInput.value = checkbox.value;
                selectedBugsContainer.appendChild(hiddenInput);
            });
            
            // Show count of selected items
            document.getElementById('bulkUpdateCount').textContent = `${checkboxes.length} گزارش انتخاب شده است`;
            
            document.getElementById('bulkUpdateModal').style.display = 'block';
        }
        
        function closeBulkUpdateModal() {
            document.getElementById('bulkUpdateModal').style.display = 'none';
        }
        
        // Response modal functions
        function showResponseModal(bugId, subject, email, name, hasResponse, existingResponse) {
            document.getElementById('responseBugId').value = bugId;
            document.getElementById('responseText').value = '';
            
            const bugInfo = `
                <div style="margin-bottom: 0.5rem;">
                    <strong style="color: var(--primary-400);">📝 موضوع:</strong>
                    <span style="color: var(--dark-300);">${subject}</span>
                </div>
                <div style="margin-bottom: 0.5rem;">
                    <strong style="color: var(--primary-400);">👤 نام:</strong>
                    <span style="color: var(--dark-300);">${name}</span>
                </div>
                <div>
                    <strong style="color: var(--primary-400);">📧 ایمیل:</strong>
                    <span style="color: var(--dark-300);">${email}</span>
                </div>
            `;
            document.getElementById('responseBugInfo').innerHTML = bugInfo;
            
            if (hasResponse && existingResponse) {
                document.getElementById('existingResponse').style.display = 'block';
                document.getElementById('existingResponseText').textContent = existingResponse;
            } else {
                document.getElementById('existingResponse').style.display = 'none';
            }
            
            document.getElementById('responseModal').style.display = 'block';
        }
        
        function closeResponseModal() {
            document.getElementById('responseModal').style.display = 'none';
            document.getElementById('responseForm').reset();
        }
        
        // Handle response form submission
        document.getElementById('responseForm').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const bugId = document.getElementById('responseBugId').value;
            const response = document.getElementById('responseText').value;
            const formType = document.getElementById('responseFormType').value;
            const submitBtn = document.getElementById('responseSubmitBtn');
            
            if (!response.trim()) {
                alert('لطفاً متن پاسخ را وارد کنید');
                return;
            }
            
            submitBtn.disabled = true;
            submitBtn.textContent = 'در حال ارسال...';
            
            try {
                const responseData = await fetch('api/send-response.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        form_type: formType,
                        id: bugId,
                        response: response
                    })
                });
                
                if (!responseData.ok) {
                    throw new Error('HTTP error! status: ' + responseData.status);
                }
                
                const data = await responseData.json();
                
                if (data.success) {
                    const message = 'پاسخ با موفقیت ارسال شد' + (data.email_sent ? ' و ایمیل به کاربر ارسال شد' : '');
                    showNotification('success', '✅ موفق', message);
                    closeResponseModal();
                    setTimeout(() => {
                        location.reload();
                    }, 2000);
                } else {
                    showNotification('error', '❌ خطا', 'خطا در ارسال پاسخ: ' + (data.message || 'خطای نامشخص'));
                }
            } catch (error) {
                console.error('Error sending response:', error);
                showNotification('error', '❌ خطا', 'خطا در اتصال به سرور');
            } finally {
                submitBtn.disabled = false;
                submitBtn.textContent = '📧 ارسال پاسخ';
            }
        });
        
        // Delete single bug
        function deleteBug(bugId) {
            if (confirm('آیا از حذف این گزارش باگ اطمینان دارید؟')) {
                const form = document.createElement('form');
                form.method = 'POST';
                form.innerHTML = `
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="bug_id" value="${bugId}">
                `;
                document.body.appendChild(form);
                form.submit();
            }
        }
        
        // Close modals when clicking outside
        window.onclick = function(event) {
            const editModal = document.getElementById('editModal');
            const viewModal = document.getElementById('viewModal');
            const bulkUpdateModal = document.getElementById('bulkUpdateModal');
            
            if (event.target === editModal) {
                closeEditModal();
            }
            if (event.target === viewModal) {
                closeViewModal();
            }
            if (event.target === bulkUpdateModal) {
                closeBulkUpdateModal();
            }
            const responseModal = document.getElementById('responseModal');
            if (event.target === responseModal) {
                closeResponseModal();
            }
        }
        
        // Close modals when pressing Escape key
        document.addEventListener('keydown', function(event) {
            if (event.key === 'Escape') {
                closeEditModal();
                closeViewModal();
                closeBulkUpdateModal();
                closeResponseModal();
            }
        });
        
        // Notification System
        function showNotification(type, title, message) {
            const container = document.getElementById('notificationContainer');
            const notification = document.createElement('div');
            notification.className = `notification notification-${type}`;
            
            const icon = type === 'success' ? '✅' : type === 'error' ? '❌' : '⚠️';
            
            notification.innerHTML = `
                <span class="notification-icon">${icon}</span>
                <div class="notification-content">
                    <div class="notification-title">${title}</div>
                    <div class="notification-message">${message}</div>
                </div>
                <button class="notification-close" onclick="this.parentElement.remove()">×</button>
            `;
            
            container.appendChild(notification);
            
            // Auto remove after 5 seconds
            setTimeout(() => {
                notification.classList.add('hiding');
                setTimeout(() => {
                    if (notification.parentElement) {
                        notification.remove();
                    }
                }, 300);
            }, 5000);
        }
        
        // Form validation
        document.getElementById('bulkUpdateForm').addEventListener('submit', function(e) {
            const status = document.getElementById('bulkStatus').value;
            const priority = document.getElementById('bulkPriority').value;
            
            if (!status && !priority) {
                e.preventDefault();
                alert('لطفاً حداقل یک فیلد را انتخاب کنید');
                return false;
            }
            
            // Show confirmation
            const checkboxes = document.querySelectorAll('.bug-checkbox:checked');
            if (!confirm(`آیا از بروزرسانی ${checkboxes.length} گزارش انتخاب شده اطمینان دارید؟`)) {
                e.preventDefault();
                return false;
            }
        });
    </script>
</body>
</html>
