<?php
header('Content-Type: application/json');
require_once 'config/database.php';

try {
    $database = new Database();
    $pdo = $database->getConnection();
    
    // Get risk threshold from settings
    $stmt = $pdo->prepare("SELECT setting_value FROM threat_detection_settings WHERE setting_key = 'risk_threshold'");
    $stmt->execute();
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    $riskThreshold = $result['setting_value'] ?? 80;
    
    // Find IPs with high risk scores
    $stmt = $pdo->prepare("
        SELECT DISTINCT ip_address, 
               COUNT(*) as threat_count,
               SUM(CASE WHEN severity = 'critical' THEN 10 
                       WHEN severity = 'high' THEN 7 
                       WHEN severity = 'medium' THEN 4 
                       ELSE 1 END) as risk_score
        FROM security_logs 
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
        GROUP BY ip_address 
        HAVING risk_score >= ?
        AND ip_address NOT IN (SELECT ip_address FROM ip_blacklist)
    ");
    $stmt->execute([$riskThreshold]);
    $suspiciousIPs = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $blockedCount = 0;
    $blockedIPs = [];
    
    if (!empty($suspiciousIPs)) {
        // Get first admin user ID for foreign key constraint
        $stmt = $pdo->query("SELECT id FROM admin_users LIMIT 1");
        $adminUser = $stmt->fetch(PDO::FETCH_ASSOC);
        $adminId = $adminUser ? $adminUser['id'] : 1; // Default to 1 if no admin found
        
        $blockStmt = $pdo->prepare("
            INSERT INTO ip_blacklist (ip_address, reason, expires_at, added_by, created_at, updated_at) 
            VALUES (?, ?, DATE_ADD(NOW(), INTERVAL 24 HOUR), ?, NOW(), NOW())
            ON DUPLICATE KEY UPDATE 
            reason = VALUES(reason), 
            expires_at = DATE_ADD(NOW(), INTERVAL 24 HOUR), 
            updated_at = NOW()
        ");
        
        foreach ($suspiciousIPs as $ip) {
            $reason = "ریسک بالا: {$ip['risk_score']} امتیاز، {$ip['threat_count']} تهدید در 24 ساعت گذشته";
            $blockStmt->execute([$ip['ip_address'], $reason, $adminId]);
            $blockedCount++;
            
            $blockedIPs[] = [
                'ip' => $ip['ip_address'],
                'risk_score' => $ip['risk_score'],
                'threat_count' => $ip['threat_count'],
                'reason' => $reason
            ];
        }
    }
    
    // Get updated statistics
    $stmt = $pdo->query("SELECT COUNT(*) as count FROM ip_blacklist");
    $totalBlocked = $stmt->fetch(PDO::FETCH_ASSOC)['count'];
    
    $stmt = $pdo->query("SELECT MAX(created_at) as last_block FROM ip_blacklist");
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    $lastBlockTime = $result['last_block'] ? date('Y-m-d H:i:s', strtotime($result['last_block'])) : 'هیچ‌وقت';
    
    // Return success response
    echo json_encode([
        'success' => true,
        'message' => "تعداد {$blockedCount} IP مشکوک مسدود شدند",
        'blocked_count' => $blockedCount,
        'risk_threshold' => $riskThreshold,
        'total_blocked' => $totalBlocked,
        'last_block_time' => $lastBlockTime,
        'blocked_ips' => $blockedIPs,
        'timestamp' => date('Y-m-d H:i:s')
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'خطا در مسدودسازی IP های مشکوک: ' . $e->getMessage()
    ]);
}
?>
