<?php
// Start output buffering to prevent any unwanted output
ob_start();

// Set JSON content type
header('Content-Type: application/json; charset=utf-8');

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

session_start();

try {
    require_once 'config/database.php';

    // Check if user is logged in
    if (!isset($_SESSION['admin_id'])) {
        ob_clean();
        http_response_code(401);
        echo json_encode(['success' => false, 'message' => 'دسترسی غیرمجاز - لطفاً دوباره وارد شوید']);
        exit();
    }

    // Check if request is POST
    if (!isset($_SERVER['REQUEST_METHOD']) || $_SERVER['REQUEST_METHOD'] !== 'POST') {
        ob_clean();
        http_response_code(405);
        echo json_encode(['success' => false, 'message' => 'متد غیرمجاز - REQUEST_METHOD: ' . ($_SERVER['REQUEST_METHOD'] ?? 'undefined')]);
        exit();
    }

    // Get POST data
    $action = $_POST['action'] ?? '';
    $ip = $_POST['ip'] ?? '';
    $reason = $_POST['reason'] ?? 'مسدودیت دستی';

    // Validate action
    if ($action !== 'block_ip') {
        ob_clean();
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'عملیات نامعتبر']);
        exit();
    }

    // Validate IP
    if (empty($ip) || !filter_var($ip, FILTER_VALIDATE_IP)) {
        ob_clean();
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'آدرس IP نامعتبر']);
        exit();
    }

    // Connect to database
    try {
        $db = new Database();
        $pdo = $db->getConnection();
    } catch (Exception $e) {
        ob_clean();
        error_log("Database connection error: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'خطا در اتصال به دیتابیس - لطفاً با مدیر سیستم تماس بگیرید']);
        exit();
    }
    
    // Check if IP is already blocked
    $stmt = $pdo->prepare("SELECT * FROM ip_blacklist WHERE ip_address = ? AND (expires_at IS NULL OR expires_at > NOW())");
    $stmt->execute([$ip]);
    $existingBlock = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($existingBlock) {
        ob_clean();
        echo json_encode(['success' => false, 'message' => 'این IP قبلاً مسدود شده است']);
        exit();
    }
    
    // Block IP for 24 hours
    $expiresAt = date('Y-m-d H:i:s', strtotime('+24 hours'));
    
    try {
        // Insert without added_by to avoid foreign key constraint issues
        $stmt = $pdo->prepare("INSERT INTO ip_blacklist (ip_address, reason, expires_at, created_at, updated_at) VALUES (?, ?, ?, NOW(), NOW())");
        $result = $stmt->execute([$ip, $reason, $expiresAt]);
        
        if (!$result) {
            $errorInfo = $stmt->errorInfo();
            error_log("INSERT error: " . print_r($errorInfo, true));
            throw new Exception("خطا در درج داده: " . $errorInfo[2]);
        }
    } catch (Exception $e) {
        error_log("INSERT exception: " . $e->getMessage());
        throw $e;
    }
    
    // Log the action
    try {
        $stmt = $pdo->prepare("INSERT INTO security_logs (event_type, ip_address, details, severity, created_at) VALUES (?, ?, ?, ?, NOW())");
        $stmt->execute(['manual_ip_block', $ip, "IP manually blocked: $reason", 'high']);
    } catch (Exception $e) {
        error_log("Security log error: " . $e->getMessage());
        // Don't fail the main operation if logging fails
    }
    
    // Clear any output buffer
    ob_clean();
    
    // Return success response
    echo json_encode([
        'success' => true, 
        'message' => 'IP با موفقیت مسدود شد',
        'ip' => $ip,
        'expires_at' => $expiresAt
    ]);
    
} catch (Exception $e) {
    // Clear any output buffer
    ob_clean();
    
    error_log("Error blocking IP: " . $e->getMessage());
    error_log("Error details: " . $e->getTraceAsString());
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'خطا در مسدود کردن IP: ' . $e->getMessage()]);
}

// End output buffering
ob_end_flush();
?>
