<?php
/**
 * Test file for upload-file.php debugging
 * فایل تست برای دیباگ upload-file.php
 */

// Enable all error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

$errors = [];
$steps = [];

try {
    $steps[] = 'Step 1: Starting test';
    
    // Test 1: Check if required files exist
    $steps[] = 'Step 2: Checking required files';
    $dbFile = __DIR__ . '/../config/database.php';
    $functionsFile = __DIR__ . '/../includes/functions.php';
    $recaptchaFile = __DIR__ . '/../config/recaptcha-config.php';
    
    if (!file_exists($dbFile)) {
        $errors[] = "Database config file not found: $dbFile";
    } else {
        $steps[] = 'Database config file exists';
    }
    
    if (!file_exists($functionsFile)) {
        $errors[] = "Functions file not found: $functionsFile";
    } else {
        $steps[] = 'Functions file exists';
    }
    
    if (!file_exists($recaptchaFile)) {
        $errors[] = "reCAPTCHA config file not found: $recaptchaFile";
    } else {
        $steps[] = 'reCAPTCHA config file exists';
    }
    
    // Test 2: Try to include files
    $steps[] = 'Step 3: Including files';
    try {
        require_once $dbFile;
        $steps[] = 'Database config included';
    } catch (Exception $e) {
        $errors[] = "Error including database.php: " . $e->getMessage();
    } catch (Error $e) {
        $errors[] = "Fatal error including database.php: " . $e->getMessage();
    }
    
    try {
        require_once $functionsFile;
        $steps[] = 'Functions file included';
    } catch (Exception $e) {
        $errors[] = "Error including functions.php: " . $e->getMessage();
    } catch (Error $e) {
        $errors[] = "Fatal error including functions.php: " . $e->getMessage();
    }
    
    try {
        require_once $recaptchaFile;
        $steps[] = 'reCAPTCHA config included';
    } catch (Exception $e) {
        $errors[] = "Error including recaptcha-config.php: " . $e->getMessage();
    } catch (Error $e) {
        $errors[] = "Fatal error including recaptcha-config.php: " . $e->getMessage();
    }
    
    // Test 3: Check if functions exist
    $steps[] = 'Step 4: Checking functions';
    if (!function_exists('getClientIP')) {
        $errors[] = "getClientIP function not found";
    } else {
        $steps[] = 'getClientIP function exists';
        try {
            $ip = getClientIP();
            $steps[] = "getClientIP returned: $ip";
        } catch (Exception $e) {
            $errors[] = "Error calling getClientIP: " . $e->getMessage();
        }
    }
    
    if (!function_exists('isIPBlacklisted')) {
        $errors[] = "isIPBlacklisted function not found";
    } else {
        $steps[] = 'isIPBlacklisted function exists';
    }
    
    if (!function_exists('checkRateLimit')) {
        $errors[] = "checkRateLimit function not found";
    } else {
        $steps[] = 'checkRateLimit function exists';
    }
    
    if (!function_exists('getSystemSetting')) {
        $errors[] = "getSystemSetting function not found";
    } else {
        $steps[] = 'getSystemSetting function exists';
    }
    
    // Test 4: Check database connection
    $steps[] = 'Step 5: Testing database connection';
    try {
        if (class_exists('Database')) {
            $db = new Database();
            $conn = $db->getConnection();
            $steps[] = 'Database connection successful';
        } else {
            $errors[] = "Database class not found";
        }
    } catch (Exception $e) {
        $errors[] = "Database connection error: " . $e->getMessage();
    } catch (Error $e) {
        $errors[] = "Database fatal error: " . $e->getMessage();
    }
    
    // Test 5: Check upload directory
    $steps[] = 'Step 6: Checking upload directory';
    $uploadDir = __DIR__ . '/../uploads/bug-reports/';
    if (!is_dir($uploadDir)) {
        $errors[] = "Upload directory does not exist: $uploadDir";
        $steps[] = "Attempting to create directory";
        if (!mkdir($uploadDir, 0755, true)) {
            $errors[] = "Failed to create upload directory";
        } else {
            $steps[] = "Upload directory created";
        }
    } else {
        $steps[] = 'Upload directory exists';
    }
    
    if (is_dir($uploadDir)) {
        if (!is_writable($uploadDir)) {
            $errors[] = "Upload directory is not writable: $uploadDir";
            $steps[] = "Directory permissions: " . substr(sprintf('%o', fileperms($uploadDir)), -4);
        } else {
            $steps[] = 'Upload directory is writable';
        }
    }
    
    // Test 6: Check PHP extensions
    $steps[] = 'Step 7: Checking PHP extensions';
    if (!function_exists('finfo_open')) {
        $errors[] = "finfo_open function not available (fileinfo extension missing)";
    } else {
        $steps[] = 'finfo_open available';
    }
    
    if (!function_exists('getimagesize')) {
        $errors[] = "getimagesize function not available";
    } else {
        $steps[] = 'getimagesize available';
    }
    
    if (!function_exists('random_bytes')) {
        $errors[] = "random_bytes function not available";
    } else {
        $steps[] = 'random_bytes available';
    }
    
} catch (Exception $e) {
    $errors[] = "Unexpected exception: " . $e->getMessage();
    $errors[] = "Stack trace: " . $e->getTraceAsString();
} catch (Error $e) {
    $errors[] = "Fatal error: " . $e->getMessage();
    $errors[] = "Stack trace: " . $e->getTraceAsString();
}

// Return results
echo json_encode([
    'success' => empty($errors),
    'errors' => $errors,
    'steps' => $steps,
    'php_version' => PHP_VERSION,
    'upload_max_filesize' => ini_get('upload_max_filesize'),
    'post_max_size' => ini_get('post_max_size'),
    'max_file_uploads' => ini_get('max_file_uploads'),
    'memory_limit' => ini_get('memory_limit')
], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
?>

