<?php
/**
 * API Endpoint for Form Configuration
 * API برای دریافت تنظیمات فرم و بررسی IP
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once '../config/database.php';
require_once '../includes/functions.php';
require_once '../config/recaptcha-config.php';

try {
    $input = json_decode(file_get_contents('php://input'), true);
    $action = $input['action'] ?? 'get_config';
    
    $db = new Database();
    $conn = $db->getConnection();
    
    if ($action === 'check_ip') {
        // Check IP blocking
        $clientIP = getClientIP();
        
        // Check if IP is whitelisted first
        if (isIPWhitelisted($clientIP)) {
            echo json_encode([
                'blocked' => false,
                'ip' => $clientIP,
                'message' => 'IP در لیست سفید قرار دارد'
            ]);
            exit();
        }
        
        // Check if IP is blacklisted
        $isBlocked = isIPBlacklisted($clientIP);
        
        if ($isBlocked) {
            $blockDetails = getBlockDetails($clientIP);
            
            echo json_encode([
                'blocked' => true,
                'ip' => $clientIP,
                'reason' => $blockDetails['reason'] ?? 'IP مسدود شده',
                'expires_at' => $blockDetails['expires_at'] ? date('Y/m/d H:i', strtotime($blockDetails['expires_at'])) : null,
                'time_remaining' => $blockDetails['expires_at'] ? getTimeRemaining($blockDetails['expires_at']) : null
            ]);
        } else {
            echo json_encode([
                'blocked' => false,
                'ip' => $clientIP,
                'message' => 'دسترسی مجاز'
            ]);
        }
    } else {
        // Get form configuration
        $recaptchaEnabled = getSystemSetting('enable_recaptcha', '1') === '1';
        $recaptchaSiteKey = $recaptchaEnabled ? RECAPTCHA_SITE_KEY : '';
        
        echo json_encode([
            'success' => true,
            'recaptcha_enabled' => $recaptchaEnabled,
            'recaptcha_site_key' => $recaptchaSiteKey
        ]);
    }
    
} catch (Exception $e) {
    error_log("Form config API error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'خطا در دریافت تنظیمات'
    ]);
}

// Helper functions
function getBlockDetails($ip) {
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        $query = "SELECT reason, expires_at, created_at FROM ip_blacklist WHERE ip_address = ? AND (expires_at IS NULL OR expires_at > NOW()) ORDER BY created_at DESC LIMIT 1";
        $stmt = $conn->prepare($query);
        $stmt->execute([$ip]);
        
        return $stmt->fetch();
    } catch (Exception $e) {
        error_log("Block details error: " . $e->getMessage());
        return null;
    }
}

function getTimeRemaining($expiresAt) {
    $expires = strtotime($expiresAt);
    $now = time();
    $diff = $expires - $now;
    
    if ($diff <= 0) {
        return 'منقضی شده';
    }
    
    $hours = floor($diff / 3600);
    $minutes = floor(($diff % 3600) / 60);
    
    if ($hours > 0) {
        return "{$hours} ساعت و {$minutes} دقیقه باقی مانده";
    } else {
        return "{$minutes} دقیقه باقی مانده";
    }
}
?>

