<?php
/**
 * API for Feature Suggestions
 * API برای دریافت پیشنهادات عمومی و رای‌دهی
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once '../config/database.php';
require_once '../includes/functions.php';

try {
    $db = new Database();
    $conn = $db->getConnection();
    
    $method = $_SERVER['REQUEST_METHOD'];
    $input = json_decode(file_get_contents('php://input'), true);
    $action = $_GET['action'] ?? $input['action'] ?? '';
    
    if ($method === 'GET' && $action === 'get_suggestions') {
        // Get all active suggestions with vote counts
        $query = "SELECT 
                    ps.id,
                    ps.title,
                    ps.description,
                    ps.is_featured,
                    ps.display_order,
                    ps.disable_vote_removal,
                    COUNT(fv.id) as vote_count,
                    ps.created_at
                  FROM public_feature_suggestions ps
                  LEFT JOIN feature_votes fv ON ps.id = fv.suggestion_id
                  WHERE ps.is_active = 1
                  GROUP BY ps.id, ps.title, ps.description, ps.is_featured, ps.display_order, ps.disable_vote_removal, ps.created_at
                  ORDER BY ps.is_featured DESC, ps.display_order ASC, vote_count DESC, ps.created_at DESC";
        
        $stmt = $conn->prepare($query);
        $stmt->execute();
        $suggestions = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Check which ones the current IP has voted for
        $clientIP = getClientIP();
        $votedIds = [];
        
        if (!empty($suggestions)) {
            $suggestionIds = array_column($suggestions, 'id');
            $placeholders = implode(',', array_fill(0, count($suggestionIds), '?'));
            $voteQuery = "SELECT suggestion_id FROM feature_votes WHERE suggestion_id IN ($placeholders) AND ip_address = ?";
            $voteStmt = $conn->prepare($voteQuery);
            $voteStmt->execute(array_merge($suggestionIds, [$clientIP]));
            $votedIds = array_column($voteStmt->fetchAll(PDO::FETCH_ASSOC), 'suggestion_id');
        }
        
        // Add voted status to each suggestion and convert is_featured to boolean
        foreach ($suggestions as &$suggestion) {
            $suggestion['has_voted'] = in_array($suggestion['id'], $votedIds);
            $suggestion['vote_count'] = (int)$suggestion['vote_count'];
            $suggestion['is_featured'] = (bool)$suggestion['is_featured']; // Ensure boolean type
        }
        unset($suggestion); // Break reference
        
        echo json_encode([
            'success' => true,
            'suggestions' => $suggestions
        ]);
        
    } elseif ($method === 'POST' && $action === 'vote') {
        // Vote for a suggestion
        $suggestionId = (int)($input['suggestion_id'] ?? 0);
        $clientIP = getClientIP();
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
        
        if (!$suggestionId) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'شناسه پیشنهاد نامعتبر است']);
            exit();
        }
        
        // Check if suggestion exists and is active
        $checkQuery = "SELECT id FROM public_feature_suggestions WHERE id = ? AND is_active = 1";
        $checkStmt = $conn->prepare($checkQuery);
        $checkStmt->execute([$suggestionId]);
        
        if (!$checkStmt->fetch()) {
            http_response_code(404);
            echo json_encode(['success' => false, 'message' => 'پیشنهاد یافت نشد']);
            exit();
        }
        
        // Check if IP has already voted
        $voteCheckQuery = "SELECT id FROM feature_votes WHERE suggestion_id = ? AND ip_address = ?";
        $voteCheckStmt = $conn->prepare($voteCheckQuery);
        $voteCheckStmt->execute([$suggestionId, $clientIP]);
        
        if ($voteCheckStmt->fetch()) {
            // Already voted - check if vote removal is disabled
            $disableCheckQuery = "SELECT disable_vote_removal FROM public_feature_suggestions WHERE id = ?";
            $disableCheckStmt = $conn->prepare($disableCheckQuery);
            $disableCheckStmt->execute([$suggestionId]);
            $suggestion = $disableCheckStmt->fetch(PDO::FETCH_ASSOC);
            
            if ($suggestion && $suggestion['disable_vote_removal']) {
                // Vote removal is disabled - return error
                http_response_code(400);
                echo json_encode([
                    'success' => false,
                    'message' => 'نمی‌توانید رای خود را پس بگیرید'
                ], JSON_UNESCAPED_UNICODE);
                exit();
            }
            
            // Vote removal is allowed - remove vote
            $deleteQuery = "DELETE FROM feature_votes WHERE suggestion_id = ? AND ip_address = ?";
            $deleteStmt = $conn->prepare($deleteQuery);
            $deleteStmt->execute([$suggestionId, $clientIP]);
            
            // Get updated vote count
            $countQuery = "SELECT COUNT(*) as vote_count FROM feature_votes WHERE suggestion_id = ?";
            $countStmt = $conn->prepare($countQuery);
            $countStmt->execute([$suggestionId]);
            $voteCount = $countStmt->fetchColumn();
            
            logSecurityEvent('feature_vote_removed', 'IP: ' . $clientIP . ', Suggestion ID: ' . $suggestionId, 'low');
            
            echo json_encode([
                'success' => true,
                'has_voted' => false,
                'vote_count' => (int)$voteCount,
                'message' => 'رای شما حذف شد'
            ], JSON_UNESCAPED_UNICODE);
        } else {
            // Add vote
            $insertQuery = "INSERT INTO feature_votes (suggestion_id, ip_address, user_agent) VALUES (?, ?, ?)";
            $insertStmt = $conn->prepare($insertQuery);
            $insertStmt->execute([$suggestionId, $clientIP, $userAgent]);
            
            // Get updated vote count
            $countQuery = "SELECT COUNT(*) as vote_count FROM feature_votes WHERE suggestion_id = ?";
            $countStmt = $conn->prepare($countQuery);
            $countStmt->execute([$suggestionId]);
            $voteCount = $countStmt->fetchColumn();
            
            logSecurityEvent('feature_vote_added', 'IP: ' . $clientIP . ', Suggestion ID: ' . $suggestionId, 'low');
            
            echo json_encode([
                'success' => true,
                'has_voted' => true,
                'vote_count' => (int)$voteCount,
                'message' => 'رای شما ثبت شد'
            ]);
        }
        
    } else {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'عملیات نامعتبر']);
    }
    
} catch (Exception $e) {
    error_log("Feature suggestions API error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'خطا در پردازش درخواست'
    ]);
}
?>

